# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import sys
import os
import subprocess
import time

import urllib
import nautilus

import gtk
import gobject
import gnomevfs

sys.path.append('/usr/bin/')
from TimeVault import config
from TimeVault.base import *
from TimeVault import dbusclient

LOGO = config.pix14x14 + '/timevault.png'

class TimeVaultNautilusProvider(nautilus.InfoProvider, nautilus.MenuProvider, nautilus.PropertyPageProvider):
	def __init__(self):
		self.cfg = Configuration()
		try:
			self.dbus = dbusclient.DBusClient()
		except:
			self.dbus = None
		# ToDo: Automatically reload config file if it changes
		
		self.tableRow = 0
		self.idleQueue = []
		self.idleSet = False
	
	# ToDo: Doesn't works due to not being able to receive messages when automated by NautilusPython
	# Is it a mainloop problem?
	#def OnNotifyConfigurationChange(self):
		#print(D_NORM, "Reloading config file")
		#self.cfg.Reload()
		
	def GetCellStats(self, cell, eventList):
		totalCount = 0
		totalSize = 0
		for event in eventList:
			try:
				totalSize += cell[event]
				totalCount += cell['#'+event]
			except:
				pass			# No events of this type
		
		return totalCount, totalSize
	
	def GetLabel(self, markup):
		L = gtk.Label()
		L.set_justify(gtk.JUSTIFY_LEFT)
		L.set_alignment(0.0, 0.5)
		L.set_padding(15, 0)
		L.set_markup(markup)
		
		return L
	
	def OnIdleGetPathInfo(self, path, bodyTable):
		x = gtk.EXPAND|gtk.FILL
		y = 0
		
		if not self.dbus.ConnectToServer():
			bodyTable.attach(gtk.Label('Cannot connect to TimeVault. Is the server running?'), 0, 2, 0, 1, xoptions=x, yoptions=y)
			return False
		
		# Stretchy header
		row = 0
		bodyTable.attach(gtk.Label(), 0, 2, row, row+1)
		row += 1
		
		longestMatch = self.cfg.LongestPathMatch(path)
		if not longestMatch:
			return False
		
		bodyTable.attach(self.GetLabel("Included in the vault as part of:"), 0, 1, row, row+1, xoptions=x, yoptions=y)
		bodyTable.attach(self.GetLabel("<b><small>'%s'</small></b>" % longestMatch), 1, 2, row, row+1, xoptions=x, yoptions=y)
		row += 1
		
		exclusion = self.cfg.Excluded(path)
		if exclusion:
			bodyTable.attach(self.GetLabel("Excluded as part of:"), 0, 1, row, row+1, xoptions=x, yoptions=y)
			bodyTable.attach(self.GetLabel("<b><small>'%s'</small></b>" % exclusion), 1, 2, row, row+1, xoptions=x, yoptions=y)
			row += 1
		
		self.finfo = {}
		eventList = 'CDNBM#>'
		for event in eventList:
			self.finfo[event] = 0
			self.finfo['#'+event] = 0
		
		try:
			stats = self.dbus.interface.Stat(path, [0, time.time()])
			for key in stats:
				self.finfo[key]= int(stats[key])
		except:
			pass				# Empty
		
		totalCount, totalSize = self.GetCellStats(self.finfo, eventList)
		bodyTable.attach(self.GetLabel("Summary:"), 0, 1, row, row+1, xoptions=x, yoptions=y)
		bodyTable.attach(self.GetLabel("<b><small>%d snapshots (including meta) (%s)</small></b>" % (totalCount, HumanSize(totalSize))), 
			1, 2, row, row+1, xoptions=x, yoptions=y)
		row += 1
		totalCount, totalSize = self.GetCellStats(self.finfo, 'BCN')
		bodyTable.attach(self.GetLabel("<b><small>Total space used on drive: (%s)</small></b>" % (HumanSize(totalSize))), 1, 2, row, row+1, xoptions=x, yoptions=y)
		row += 1
		
		# Stretchy partition
		bodyTable.attach(gtk.Label(), 0, 2, row, row+1)
		row += 1
		
		for event in eventList:
			bodyTable.attach(self.GetLabel("<b><small>%s</small></b>:" % EVENTNAME[event]), 0, 1, row, row+1, xoptions=x, yoptions=y)
			bodyTable.attach(self.GetLabel("<b><small>%d files (%s)</small></b>" % (self.finfo['#'+event], HumanSize(self.finfo[event]))), 1, 2, row, row+1, xoptions=x, yoptions=y)
			row += 1
			
		# Stretchy footer
		bodyTable.attach(gtk.Label(), 0, 2, row, row+1)
		row += 1
		
		bodyTable.show_all()
		return False
		
	def LaunchSnapshotBrowser(self, widget, path, asRoot):
		if asRoot:
			cmd = 'gksudo --preserve-env "%s/timevault-browser --path=\'%s\'"' % (config.bindir, path)
		else:
			cmd = '%s/timevault-browser --path=\'%s\'' % (config.bindir, path)
		
		sub = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE)
	
	def get_property_pages(self, files):
		if len(files) != 1:
			return
		
		file = files[0]
		if file.get_uri_scheme() != 'file':
			return

		path = urllib.unquote(file.get_uri()[7:])
		if file.is_directory():
			path += '/'

		if not self.cfg.LongestPathMatch(path):
			return
		
		self.vbox = gtk.VBox()
		
		self.bodyTable = gtk.Table(rows=12, columns=2)
		self.vbox.pack_start(self.bodyTable, True, True, 10)
		
		buttonBox = gtk.HButtonBox()
		
		buttonLogo = gtk.Image()
		buttonLogo.set_from_file(LOGO)
		button = gtk.Button('Launch Snapshot Browser')
		button.set_image(buttonLogo)
		button.connect('clicked', self.LaunchSnapshotBrowser, path, False)
		buttonBox.pack_start(button)
		
		buttonLogo = gtk.Image()
		buttonLogo.set_from_file(LOGO)
		button = gtk.Button('Launch Snapshot Browser as Root')
		button.set_image(buttonLogo)
		button.connect('clicked', self.LaunchSnapshotBrowser, path, True)
		buttonBox.pack_start(button)
		
		self.vbox.pack_start(buttonBox, False, False, 10)
		
		self.logo = gtk.Image()
		self.logo.set_from_file(LOGO)
		self.logo.show()
		
		self.vbox.show_all()
		gobject.idle_add(self.OnIdleGetPathInfo, path, self.bodyTable)
		return nautilus.PropertyPage("NautilusPython::timeVault", self.logo, self.vbox),
	
	def OnIdle(self):
		return False
	
	def update_file_info(self, file):
		if file.get_uri_scheme() != 'file':
			return
		path = gnomevfs.get_local_path_from_uri(file.get_uri())

		if file.is_directory():
			path += '/'
		
		if self.cfg.LongestPathMatch(path)==path:
			file.add_emblem('urgent')

	def get_toolbar_items(self, window, file):
		if file.get_uri_scheme() != 'file':
			return

		path = urllib.unquote(file.get_uri()[7:])
		if not file.is_directory():
			return
		else:
			path += '/'
		
		# ToDo: Post to pynautilus mailing list and ask them how to add OnNotifyConfigurationChange
		self.cfg.Reload()
		if not self.cfg.LongestPathMatch(path):
			return
		
		self.window = window
		menuItem = nautilus.MenuItem("NautilusPython::timeVault", "TimeVault", "Browse previous versions with TimeVault", config.pix32x32+'/timevault.png')
		menuItem.connect("activate", self.OnMenuIconClicked, path)

		return menuItem, 

	def OnMenuIconClicked(self, arg, path):
		self.LaunchSnapshotBrowser(None, path, False)
