# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import os
import statvfs
import sys
import stat
import time
import re
import getopt
import traceback
import subprocess
from pprint import PrettyPrinter

try:
	import hashlib			# New in Python 2.5
except:
	sys.stderr.write("Could not load hashlib (Python version <2.5?), using sha instead\n")
	import sha				# Deprecated (in 2.5) method

import config

# Version Information
TIMEVAULT_CONFIGURATION_FILE_VERSION = '1.4'

# Path definitions

# Trailing slash required for all directory entries
NAUTILUS_PLUGIN_PATH = '/usr/lib/nautilus/extensions-1.0/python/'

DEFAULT_ROOT = '/backup/'
DIR_CATALOG = 'catalog/'
DIR_INTERNAL = 'internal/'
DIR_PENDING =  'pending/'
DB_FILENAME = 'catalog.db'

DATESTAMP_FORMAT = '%Y-%m-%d'
TIMESTAMP_FORMAT = '%H-%M-%S'
DATETIME_STAMP = DATESTAMP_FORMAT+'.'+TIMESTAMP_FORMAT
EXT_INFO = '.info'
EXT_LOG = '.log'

DEFAULT_CONFIGURATION_FILE = config.confdir + '/timevault.conf'
DEFAULT_LOG_PATH = '/var/log/'
DEFAULT_LOCK_FILE = '/var/run/timevault.pid'

# Operational definitions

# Debug levels
D_NORM = 1
D_VERB = 2
D_ALL  = 3

# States
TVS_IDLE = 0
TVS_INFO = 1
TVS_WARN = 2
TVS_ERROR = 3

# Icons
TVI_NORM = config.media+'/timevault.png'
TVI_RECONNECT = config.media+'/reconnect.png'
TVI_PENDING = config.media+'/pending.png'
TVI_UNCONFIGURED = config.media+'/unconfigured.png'
TVI_ERROR = config.media+'/error.png'

BLOCKSIZE = 4096
EVENTNAME = {'C':'Changed', 'N':'Created', 'B':'Baseline', 
	'D':'Deleted', 'M':'Metadata', '#': 'Drivespace Abort', '>': 'Filesize Abort'}
REALEVENTS = 'BCN'
METAEVENTS = 'DM#>'
WHITESPACE = ' \t\r\n'

SIZE1K = 1024
SIZE1M = SIZE1K*SIZE1K
SIZE1G = SIZE1M*SIZE1K
REALLY_BIG = 640*SIZE1M*SIZE1M													# No backup should be this large -
																				# 640TB should be large enough for anyone ;)
TIME1M = 60
TIME1H = 3600
TIME1D = TIME1H*24

class Configuration:
	def __init__(self, filename = None):
		if not filename:
			filename = DEFAULT_CONFIGURATION_FILE
		self.filename = filename
		self.Reload()
	
	def Reload(self):
		self.Default()
		self.Read()
		
		# Always ensure that we cannot recursively back ourselves up:
		# Backup of the backup of the backup of the backup of the backup ...
		if self.GetDirBase() not in self.settings['exclude']:
			self.settings['exclude'].append(self.GetDirBase())
			
		self.exreg = self.ConvertRegEx(self.settings['exclude'])
		self.backsort = self.SortDictonaryKey(self.settings['backup'])
	
	def Default(self):
		self.settings = {}
		
		self.settings['configValid'] = False
		self.settings['tvVersion'] = config.VERSION
		self.settings['cfgVersion'] = TIMEVAULT_CONFIGURATION_FILE_VERSION
		self.settings['enabled'] = False
		self.settings['showNotifications'] = True
		self.settings['ticInc'] = 60
		self.settings['baselines'] = False
		
		self.settings['root'] = DEFAULT_ROOT
		self.settings['backup'] = {'/etc/':{'delay':60}}
		self.settings['exclude'] = ['**/lost+found/', '**core', '**~', '**.nfs*', '/tmp/**', '/var/**', 
			'/proc/**', '/mnt/**', '/dev/**', '/sys/**', '/home/*/.Trash/**', '/home/*/.beagle/**', 
			'/home/*/.Tracker/**', '/home/*/.mozilla**', '/home/*/.thumbnails/**']
		
		self.settings['reserved'] = '1GB'
		self.settings['max-size'] = '32MB'
		self.settings['rescheduleAggressiveness'] = 100			# How many to process at a time when restoring
		self.settings['deferDelay'] = 100						# How long to wait before issuing call when deferment is requested (ms)
		self.settings['snapshotNotificationCount'] = 5			# The number of files to list when initiating a snapshot balloon
		self.settings['bipTimer'] = 10000						# Debug tic timer (ms)
		self.settings['serverStatCachingEnabled'] = False
		
		self.settings['nautilusIntegration'] = True
		self.settings['diableAutoMetadata'] = False
		
		self.settings['retain'] = {'apply': ['time', 'snaps', 'bytes'], 'time': '7 days', 'snaps': 10, 'bytes': '32MB'}
		self.settings['expire'] = {'%': 10, 'period': 3600, 'apply': ['meta', 'merge', 'thin'], 'meta': True, 'merge': '1 hour', 'thin': '1 days', 'spread': 4}
	
	def Read(self):
		Debug(D_NORM, "TimeVault v.%s Reading configuration v.%s [%s]\n" % (config.VERSION, TIMEVAULT_CONFIGURATION_FILE_VERSION, DEFAULT_CONFIGURATION_FILE))
		try:
			settings = ParseRawDataStructsFile(open(self.filename, 'r').read())
			self.settings.update(settings)
			self.cfgFileFound = self.settings['configValid']
		except:
			self.settings['configValid'] = False
			self.Write()
			Debug(D_NORM, "Configuration not found - creating default\n")
			return False
		
		return True

	def Write(self):
		PrettyPrinter(indent=2)
		pp = PrettyPrinter(indent=2)
		configuration = pp.pformat(self.settings)+"\n"
		open(self.filename, 'w').write(configuration)
		
		return True
	
	def Excluded(self, path):
		for regex,pattern in self.exreg:
			if regex.match(path):
				Debug(D_ALL, "Excluding: '%s' based on '%s'\n" % (path, pattern))
				return pattern
		return False
	
	def SortDictonaryKey(self, D):
		L = []
		for K in D:
			L.append(K)
		L.sort()
		L.reverse()
		return L

	def ConvertRegEx(self, exclude):
		exreg = []
		for pattern in self.settings['exclude']:
			if not pattern:
				continue
			pattern = re.escape(pattern)
			pattern = pattern.replace('\\?', '.')				# Match a single character
			pattern = pattern.replace('\\*\\*', '.*')			# Match any charactes
			pattern = pattern.replace('\\*', '[^/]*')			# Match until '/'
			pattern = '^' + pattern + '$'						# Match from beginning until end
			
			exreg.append([re.compile(pattern), pattern])
			Debug(D_VERB, "Exclusion pattern compiled: '%s'\n" % pattern)
		return exreg

	def LongestPathMatch(self, pathFilter):
		# Remember, backsort is reverse sorted
		M = len(pathFilter)
		for path in self.backsort:
			L = len(path)
			if L>M:	
				continue		# pathFilter is not as deep as backupDir
			if pathFilter[:L]==path:
				return path		# longest match found
		return None
	
	# Directory Structure:
	'''
	/backup/										<= 755
		catalog/									<= 755
		internal/									<= 755 (this would be set to 700 if we were to remount as ro)
			uid-0/									<= 700
				2006-06-16/								<= 755
					568ade9dbae400614f882128982d2e32e032949a.tv.doc.txt
	'''
	
	# All directory functions return with a terminal '/'
	def GetDirBase(self):
		return self.settings['root']
	
	def GetDirCatalog(self):
		return self.GetDirBase()+DIR_CATALOG

	def GetDirPending(self):
		return self.GetDirBase()+DIR_PENDING

	def GetDirInternal(self):
		return self.GetDirBase()+DIR_INTERNAL
	
	def GetDirUser(self, uid):
		return self.GetDirInternal() + 'uid-%d/' % uid
	
	def GetDirSnapDest(self, uid, tm):
		return self.GetDirUser(uid)+time.strftime(DATESTAMP_FORMAT, tm)+"/"
	
	# Returns True if a path characteristic has changed, False otherwise
	def CheckAndMake(self, path, rights, uid=0):
		changed = False
		try:
			if not os.access(path, os.X_OK):
				os.makedirs(path, rights)
				os.chown(path, uid, uid)
				Debug(D_NORM, "Creating new directory: '%s'\n" % path)
				return True
		except:
			return False
		
		s = os.stat(path)
		try:
			if stat.S_IMODE(s.st_mode)!=rights:
				os.chmod(path, rights)
				Debug(D_NORM, "Mode changed for directory: '%s'\n" % path)
				changed = True
			if s.st_uid!=uid:
				os.chown(path, uid, uid)
				Debug(D_NORM, "Owner changed for directory: '%s'\n" % path)
				changed = True
		except:
			return changed
		return changed
	
	def MakePaths(self):
		remount = False
		if self.CheckAndMake(self.GetDirBase(), 0755):
			remount = True
		if self.CheckAndMake(self.GetDirInternal(), 0755):
			remount = True
		if self.CheckAndMake(self.GetDirCatalog(), 0755):
			remount = True
		if self.CheckAndMake(self.GetDirPending(), 0755):
			remount = True

	def FreeSpaceOnBackupDrive(self):
		# Backup directory may be on another drive than the others - that's ok.
		# It's the one we will store data in, so it limits operations, not the others
		try:
			s = os.statvfs(self.GetDirBase())
			return int(s[statvfs.F_BAVAIL] * s[statvfs.F_BSIZE])
		except:
			return 0
	
	def UsableSpaceOnBackupDrive(self):
		reserved = ParseSize(self.settings['reserved'])
		available = self.FreeSpaceOnBackupDrive()
		
		if reserved>available:
			return 0
		return available-reserved
	
	def FileStamp(self, path, tm):
		# Time is in unixTime
		head, tail = os.path.split(path)
		try:
			m = hashlib.sha1()
		except:
			m = sha.new()
			
		m.update("%s.%d" % (path,tm))
		return m.hexdigest()+".tv."+tail
	
		# ToDo: Using base64 encoding allows a fallback method to decode
		# filenames in the event of lost database or dead daemon
		# The problem is that the pathname might be too long
		#head, tail = os.path.split(path)
		#return '%s.%d.%s' % (base64.urlsafe_b64encode(head), tm, tail)
	
	def SnapshotPath(self, path, tm, uid):
		return self.GetDirSnapDest(uid, time.localtime(tm)) + self.FileStamp(path, tm)

# Returns command dictionary for accepted commands
DEBUG = 0
LOG_HANDLE = None
BENCH = False
FILE_LOGGING = None

def ParseCommandLineOptions():
	global DEBUG
	global BENCH
	commands = {}
	
	try:
		opts, args = getopt.getopt(sys.argv[1:], "dc:svb:e:p:r:n:", ["bench", "config=", "simulate", "verbose", "start=", "stop=", "path=", "ratio=", "number="])
	except getopt.GetoptError:
		sys.exit(2)
		
	commands = {'configFile':None, 'simulate':False, 'verbose':False}
	for opt, arg in opts:
		if opt in ("-c", "--config"):
			commands['configFile'] = arg
		elif opt in ("-p", "--path"):
			commands['path'] = arg
		elif opt in ("-d", "--bench"):
			BENCH = True
		elif opt in ("-s", "--simulate"):
			commands['simulate'] = True
		elif opt in ("-v", "--verbose"):
			commands['verbose'] = True
			DEBUG += 1					# Call verbose multiple times for extra debug info
		elif opt in ("-b", "--start"):
			commands['start'] = float(arg)
		elif opt in ("-e", "--stop"):
			commands['stop'] = float(arg)
		elif opt in ("-r", "--ratio"):
			commands['ratio'] = float(arg)
		elif opt in ("-n", "--number"):
			commands['number'] = int(arg)
	
	return commands

def Verbosity():
	global DEBUG
	return DEBUG

def SetFileLogging(filename):
	global FILE_LOGGING
	FILE_LOGGING = filename
	
def Debug(verbosity, msg, printTraceback=False, timeStamp=True):
	global DEBUG, LOG_HANDLE, FILE_LOGGING
	
	if verbosity<=DEBUG:
		if printTraceback:
			msg += traceback.format_exc()
		if timeStamp:
			tmStr = time.strftime("%b %d %H:%M:%S\t", time.localtime())
			sys.stdout.write(tmStr)
		sys.stdout.write(msg)
		if FILE_LOGGING:
			try:
				if LOG_HANDLE==None:
					if os.access(FILE_LOGGING, os.W_OK):
						LOG_HANDLE = open(FILE_LOGGING, 'a')
					elif os.geteuid()==0:
						# Open for writing, not append
						LOG_HANDLE = open(FILE_LOGGING, 'w')
				if timeStamp:
					LOG_HANDLE.write(tmStr)
				LOG_HANDLE.write(msg)
				if verbosity<D_VERB:
					LOG_HANDLE.flush()
			except:
				sys.stdout.write("Cannot open system log: '%s'\n\tProceeding to freak out quietly\nReason Follows:" % FILE_LOGGING)
				sys.stdout.write(traceback.format_exc())
				LOG_HANDLE = open('/dev/null', 'w')

def Bench(name, tm, cnt):
	global BENCH
	if not BENCH:
		return
	
	diff = time.time() - tm
	if cnt:
		Debug(D_NORM, "Bench(%s): %d count, %f sec, %f ms/per\n" % (name, cnt, diff, 1000*diff/cnt))
	else:
		Debug(D_NORM, "Bench(%s): %d count, %f sec\n" % (name, cnt, diff))
	
def Spawn(cmd):
	sub = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE)
	
	while True:
		line = sub.stdout.readline()
		if not line:
			break
		yield line

def Exec(cmd):
	pipe = os.popen2(cmd)[1]
	
	while True:
		line = pipe.readline()
		if not line:
			break
		yield line

def ParseSize(line):
	line = line.strip()
	if line=='':
		return 0
	
	reHumanSize = re.compile('([\d\.]+)\s*(.)[Bb]')
	m = reHumanSize.match(line)
	if m:
		g = m.groups()
		s = float(g[0])
		if len(g)<2:
			return int(s)
		
		scale = g[1].upper()
		if scale=='K':
			return s*SIZE1K
		elif scale=='M':
			return s*SIZE1M
		elif scale=='G':
			return s*SIZE1G
	
	# If no units are provided or there is a B at the end, assume just bytes
	reHumanSizeUnspecified = re.compile('([\d]+)\s*[Bb]*')
	m = reHumanSizeUnspecified.match(line)
	if m:
		g = m.groups()
		return int(g[0])
	
	# Fail safe
	return REALLY_BIG

def HumanSize(sz):
	sz = float(sz)
	if sz > SIZE1G:
		return "%.1f GB" % (sz/SIZE1G)
	elif sz > SIZE1M:
		return "%.1f MB" % (sz/SIZE1M)
	elif sz > SIZE1K:
		return "%.1f KB" % (sz/SIZE1K)
	return "%d B" % sz

def HumanTime(tm):
	tm = float(tm)
	if int(tm/60)==0:
		return "%.1f secs" % (tm)
	elif int(tm/3600)==0:
		return "%.1f mins" % (tm/60)
	elif int(tm/(3600*24))==0:
		return "%.1f hours" % (tm/3600)
	return "%.1f days" % (tm/(3600*24))

def ParseTime(line):
	line = line.strip()
	if line=='':
		return 0
	
	reHumanTime = re.compile('([\d\.]+)\s*(.)')
	m = reHumanTime.match(line)
	if m:
		g = m.groups()
		s = float(g[0])
		if len(g)<2:
			return int(s)
		
		units = g[1].upper()
		if units[0]=='S':
			return s
		elif units[0]=='M':
			return s*60
		elif units[0]=='H':
			return s*60*60
		elif units[0]=='D':
			return s*60*60*24
		elif units[0]=='W':
			return s*60*60*24*7
	
	return 900			# Assume 900 secs

# Returns True for the child and False for the parent
def Daemonize(stdin='/dev/null', stdout='/dev/null', stderr='/dev/null'):
	try:
		if os.fork():
			return False			# Caller
	except:
		print("Could not fork")
		return False				# Failed to fork

	os.setsid()						# Become session leader
	os.chdir('/')					# Allow unmounting when running
	os.umask(0)						# Clear masks
	
	try:
		if os.fork():
			os._exit(0)				# First child
	except:
		os._exit(0)					# Too late to return
	
	os.close(sys.stdin.fileno())
	os.close(sys.stdout.fileno())
	os.close(sys.stderr.fileno())
	
	stdin_f = file(stdin, 'r')
	stdout_f = file(stdout, 'a+')
	stderr_f = file(stderr, 'a+', 0)
	
	os.dup2(stdin_f.fileno(), sys.stdin.fileno())
	os.dup2(stdout_f.fileno(), sys.stdout.fileno())
	os.dup2(stderr_f.fileno(), sys.stderr.fileno())
	
	return True						# Second child

def ParseRawDataStructsFile(source):
	try:
		import configparser as parserLib
		obj, rest = parserLib.parseOne(source)
		return obj
	except:			# Using Python <2.5?
		if sys.version_info[0]<=2 and sys.version_info[1]<5:
			sys.stderr.write("Could not load config parser (Python version <2.5?), using something else instead\n")
			# Check the actual version in case it was just a bad file (which 
			# we oviously don't want to pass to eval
			return eval(source)

#############################################################################################
if __name__ == "__main__":
	cfg = Configuration()
	
	pp = PrettyPrinter(indent=2)
	pp.pprint(cfg.settings)
