# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import sys
import os
import time
import math
import subprocess
import shutil

import pygtk
pygtk.require("2.0")
import gobject
import gtk
import gtk.glade

import pango
import mimetypes

# ToDo: Make boxes and file list update in realtime

# Add the appropriate path extension so we can import modules without changing directories
sys.path.append(os.path.dirname(__file__))
import config
from base import *
import database
import dbusclient

'''Helper Dialogs'''
def Message(window, markup):
	dlg = gtk.MessageDialog(parent=window, flags=gtk.DIALOG_MODAL, type=gtk.MESSAGE_WARNING, buttons=gtk.BUTTONS_OK)
	dlg.set_markup(markup)
	dlg.run()
	dlg.destroy()
	del dlg

def Ask(window, markup):
	dlg = gtk.MessageDialog(parent=window, flags= gtk.DIALOG_MODAL, type=gtk.MESSAGE_WARNING, buttons=gtk.BUTTONS_OK_CANCEL)
	dlg.set_markup(markup)
	ret = dlg.run()
	dlg.destroy()
	del dlg
	if ret==gtk.RESPONSE_OK:
		return True
	return False

def SelectDirectory(title='Please select a directory'):
	dlg = gtk.FileChooserDialog(title, None, gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER, 
		(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, gtk.STOCK_OPEN, gtk.RESPONSE_OK))
	if dlg.run() == gtk.RESPONSE_OK:
		directory = dlg.get_filename() + '/'
	else:
		directory = None

	dlg.destroy()
	return directory

MINUTE=60
HOUR=MINUTE*60
DAY=HOUR*24
WEEK=DAY*7
MONTH=DAY*30

ZOOM = [[MINUTE, MINUTE/12], [HOUR, HOUR/12], [DAY/2, HOUR], [WEEK, DAY/2], [WEEK*8, WEEK], [MONTH*24, MONTH*2]]
ZOOM_MIN_SCALE = 1
ZOOM_HOUR = 1
ZOOM_HDAY = 2
ZOOM_WEEK = 3
ZOOM_MONTH = 4

class Picker(dbusclient.DBusClient):
	def __init__(self):
		dbusclient.DBusClient.__init__(self)
		
		self.cfg = Configuration()
		self.uid = os.geteuid()
		self.commands = ParseCommandLineOptions()
		
		self.boxhbox = None
		self.selectedRidge = None
		self.selectedTmRng = None
		self.tm = None
		self.zoom = None
		self.path = '/'
		self.hideMeta = False
		self.hideReal = False
		self.treeCursor = None
		self.cell = {}
		self.color = {'B':gtk.gdk.color_parse("blue"), 'N':gtk.gdk.color_parse("darkblue"), 'D':gtk.gdk.color_parse("black"), 
			'C':gtk.gdk.color_parse("green"), 'M':gtk.gdk.color_parse("white"), '#':gtk.gdk.color_parse("red"), 
			'>':gtk.gdk.color_parse("red"), 'selected':gtk.gdk.color_parse("darkgray"), 'unselected':gtk.gdk.color_parse("navajowhite")}
		
		self.smallFont = pango.FontDescription('Sans condensed 6')
		self.mediumFont = pango.FontDescription('Sans 8')
		
		# Load the glade file
		gladeFile = config.pkgdatadir+"/browser.glade"
		
		self.widgets = gtk.glade.XML(gladeFile)
		self.widgets.signal_autoconnect(self)
		
		# Connect to main window
		self.window = self.widgets.get_widget("windowPicker")
		self.cellContainer = self.widgets.get_widget("eventboxCellContainer")
		self.cellContainer.set_size_request(640, -1)

		# Treeview
		headers = ['Time', 'Path', 'Event', 'Size']
		
		# ListStore: 				id, tm(unix), size(B), time, path, event, size
		#							0   1         2        3     4     5      6
		self.listStore = gtk.ListStore(int, int, int, str, str, str, str)
		self.treeView = self.widgets.get_widget('treeviewFiles')
		self.treeView.set_size_request(-1, 320)
		self.treeView.modify_font(self.mediumFont)
		self.treeView.set_model(self.listStore)
		self.treeView.set_reorderable(True)
		self.treeView.connect("button-press-event", self.TreeviewClicked, None)
		self.treeView.connect("key-release-event", self.TreeviewKeyPressed, None)
		treeSelection = self.treeView.get_selection()
		treeSelection.set_mode(gtk.SELECTION_MULTIPLE)
		
		# Just text rendering for the columns
		textrenderer = gtk.CellRendererText()
		
		# Add the columns to the treeview
		column = gtk.TreeViewColumn('Time', textrenderer, text=3)
		column.set_sort_column_id(1)
		column.set_resizable(True)
		self.treeView.append_column(column)

		column = gtk.TreeViewColumn('Path', textrenderer, text=4)
		column.set_sort_column_id(4)
		column.set_resizable(True)
		self.treeView.append_column(column)

		column = gtk.TreeViewColumn('Event', textrenderer, text=5)
		column.set_sort_column_id(5)
		column.set_resizable(True)
		self.treeView.append_column(column)

		column = gtk.TreeViewColumn('Size', textrenderer, text=6)
		column.set_sort_column_id(2)
		column.set_resizable(True)
		self.treeView.append_column(column)

		self.MakeFilterMenu()
		self.MakeBoxMenu()
		
		self.boxHeight = 100
		mimetypes.init()
		
	def MakeBoxMenu(self):
		self.boxmenu = gtk.Menu()
		
		deli = gtk.ImageMenuItem("Delete all")
		deli.connect('activate', self.DeleteAll)
		
		self.boxmenu.add(deli)
		self.boxmenu.show_all()
		
	def MakeFilterMenu(self):
		self.filtermenu = gtk.Menu()
		
		deli = gtk.ImageMenuItem("Filter on this file")
		deli.connect('activate', self.FilterOnFile)
		
		self.filtermenu.add(deli)
		self.filtermenu.show_all()
		
	def TreeviewClicked(self, widget, event, data):
		if event.type == gtk.gdk._2BUTTON_PRESS and event.button == 1:		# Double left-click
			self.OnButtonOpenClicked()
		elif event.button == 3:		# Right-click
			self.filtermenu.popup(parent_menu_shell=None, parent_menu_item=None, func=None, button=event.button, activate_time=event.time, data=None)

	def TreeviewKeyPressed(self, widget, event, data):
		if event.keyval == gtk.keysyms.Return:
			self.OnButtonOpenClicked()
		elif event.keyval == gtk.keysyms.Delete:
			self.OnButtonDeleteClicked()
	
	def EventsOfInterest(self):
		eventList = ''
		if not self.hideMeta:
			eventList += 'DBM#>'
		if not self.hideReal:
			eventList += 'CBN'
		return eventList

	def GetCellStats(self, cell):
		eventList = self.EventsOfInterest()
		
		totalCount = 0
		totalSize = 0
		for event in eventList:
			totalCount += cell['#'+event]
			totalSize += cell[event]
			
		if totalSize>=1024:
			totalPixels = int(2 * math.log10(totalSize/1024) / math.log10(1.5))
		else:
			totalPixels = 0
		
		return totalCount, totalSize, totalPixels
	
	def FilterOnFile(self, menuItem):
		selection = self.GetSelectedPaths()
		if not selection:
			return
		rid,rpath,rtm,revent = selection[0]					# Just the first one
		
		self.cell = {}
		self.UpdateBoxes(path=rpath)
		if self.selectedRidge:
			self.FillTree(self.selectedTmRng)
	
	def OnSnap(self, tm, num, firstN):
		Debug(D_NORM, "OnSnap: %d\n" % tm)
		tmRngRefresh = []
		for tmRng in self.cell:
			ref = self.cell[tmRng]
			if ref['start']<=tm and tm<ref['stop']:
				Debug(D_NORM, "OnSnap flush: %s\n" % tmRng)
				
				tmRngRefresh.append(tmRng)
		
		if len(tmRngRefresh):
			for tmRng in tmRngRefresh:
				del self.cell[tmRng]
			
			self.UpdateBoxes()
			if self.selectedRidge:
				self.FillTree(self.selectedTmRng)
		
	def DeleteAll(self, menuItem):
		# ToDo: Create cell-cache invalidation function that searches based on timepoint so multiple-levels 
		# can be invalidated simultaneously, replace 'del self.cell[self.selectedTmRng]
		
		ref = self.cell[self.selectedTmRng]
		startS = time.strftime("%X %x", time.localtime(ref['start']))
		stopS = time.strftime("%X %x", time.localtime(ref['stop']))
		totalCount, totalSize, totalPixels = self.GetCellStats(ref)
		
		msg = "You are about to delete %d files (%s) from\n%s-%s\n\n" % (totalCount, HumanSize(totalSize), startS, stopS)
		msg += "<b><small>Created:\t\t%d files (%s)\nChanged:\t%d files (%s)\nDeleted:\t\t%d files (%s)\nMeta:\t\t%d files (%s)</small></b>" % (ref['#N']+ref['#B'], HumanSize(ref['N']+ref['B']), ref['#C'], HumanSize(ref['C']), ref['#D'], HumanSize(ref['D']), ref['#M'], HumanSize(ref['M']))
		if not Ask(self.window, msg):
			return
		
		eventList = ''
		if self.hideMeta or self.hideReal:
			eventList = self.EventsOfInterest()
			if not len(eventList):
				return					# Apparently, we don't want to see any events
		
		try:
			self.interface.DeleteQuery(self.path, (ref['start'], ref['stop']), eventList)
		except:		# Empty or no server
			pass
		
		del self.cell[self.selectedTmRng]
		self.UpdateBoxes()
		if self.selectedRidge:
			self.selectedRidge = None
	
	def GetSelectedPaths(self):
		try:
			selection = []
			(model, pathlist) = self.treeView.get_selection().get_selected_rows()
			for treePath in pathlist:
				treeIter = self.listStore.get_iter(treePath)
				
				rid = self.listStore.get_value(treeIter, 0)
				rtm = self.listStore.get_value(treeIter, 1)
				rpath = self.listStore.get_value(treeIter, 4)
				revent = self.listStore.get_value(treeIter, 5)
				
				selection.append((rid, rpath, rtm, revent))
				
				Debug(D_VERB, "Selection: '%s'\n" % rpath)
			return selection
		except:
			Debug(D_NORM, "No selection\n", printTraceback=True)
			return None
		
	def OnButtonOpenClicked(self, *args):
		selection = self.GetSelectedPaths()
		if not selection:
			return
		rid,rpath,rtm,revent = selection[0]					# Just the first one
		snapshotPath = self.cfg.SnapshotPath(rpath, rtm, self.uid)

		sub = subprocess.Popen(['/usr/bin/gnome-open', "%s" % snapshotPath])
	
	def IsText(self, path):
		if not os.access(path, os.R_OK):
			return False
		
		(etype, encoding) = mimetypes.guess_type(path)
		if etype and etype[:4]=='text':
			return True
		
		# Otherwise, we have to do this the hard way
		try:
			fd = os.open(path, os.O_RDONLY)
			buf = os.read(fd, BLOCKSIZE)
			if not buf:
				return False
			
			bufLen = len(buf)
			nonPrintables = 0
			for i in range(bufLen):
				a = ord(buf[i])
				if (a<8) or (a>13 and a<32) or (a>126):
					nonPrintables += 1
			Debug(D_VERB, "Non-printables in %s: %d/%d\n" % (path, nonPrintables, bufLen))
			if nonPrintables<(len(buf)/10):
				return True
		except:
			Debug(D_NORM, "Could not identify file mode: %s\n" % path, printTraceback=True)
			
		return False
	
	def GetFilesToDiff(self, selection):
		if len(selection)==1:
			# Compare snapshot to current
			rid,rpath,rtm,revent = selection[0]
			snapshotPath = self.cfg.SnapshotPath(rpath, rtm, self.uid)

			textmode = self.IsText(snapshotPath)		# We will check the snapshot since 
														# the original may no longer be there
			return snapshotPath, rpath, textmode
		elif len(selection)==2:
			# Compare 2 snapshots
			rid1,rpath1,rtm1,revent1 = selection[0]
			rid2,rpath2,rtm2,revent2 = selection[1]
			
			if rpath1==rpath2:
				snapshotPath1 = self.cfg.SnapshotPath(rpath1, rtm1, self.uid)
				snapshotPath2 = self.cfg.SnapshotPath(rpath2, rtm2, self.uid)
				
				textmode = self.IsText(snapshotPath1)
				return snapshotPath1, snapshotPath2, textmode
		
		return None, None, False
	
	def OnButtonDiffClicked(self, *args):
		selection = self.GetSelectedPaths()
		if not selection:
			return
		
		path1, path2, textmode = self.GetFilesToDiff(selection)
		if path1 and path2 and textmode:
			sub = subprocess.Popen(['/usr/bin/meld', path1, path2])
		else:
			sub1 = subprocess.Popen(['/usr/bin/gnome-open', path1])
			sub2 = subprocess.Popen(['/usr/bin/gnome-open', path2])

	def IdlyHandleDiffButton(self):
		# PyGTK-needed work-around for correct response to multiselect
		selection = self.GetSelectedPaths()
		if (not selection) or (len(selection)==0) or (len(selection)>2):
			self.widgets.get_widget('buttonDiff').hide()
			return False
		
		ER = 3				# Event row in returned selection
		notDeletions = []
		for row in selection:
			if row[ER] not in 'BCN':
				continue
			notDeletions.append(row)
		
		path1, path2, textmode = self.GetFilesToDiff(selection)
		if path1 and path2 and textmode:
			self.widgets.get_widget('buttonDiff').show()
		else:
			self.widgets.get_widget('buttonDiff').hide()
		
		return False
		
	def OnTreeviewRowActivated(self, *args):
		gobject.idle_add(self.IdlyHandleDiffButton)
	
	def FillTree(self, tmRng):
		if not tmRng:
			return
		
		if self.selectedRidge:
			self.selectedRidge.modify_bg(gtk.STATE_NORMAL, self.color['unselected'])
			self.selectedRidge = None
		
		self.selectedTmRng = tmRng
		ref = self.cell[tmRng]
		ridge = ref['ridge']
		self.selectedRidge = ridge
		ridge.modify_bg(gtk.STATE_NORMAL, self.color['selected'])
		
		eventList = ''
		if self.hideMeta or self.hideReal:
			eventList = self.EventsOfInterest()
			if not len(eventList):
				return					# Apparently, we don't want to see any events
		try:
			self.treeCursor = self.interface.Find(self.path, (ref['start'], ref['stop']), eventList)
		except:		# Empty or no server
			self.treeCursor = None
		
		self.listStore.clear()
		self.OnTreeviewRowActivated()
		gobject.idle_add(self.OnIdleAddRow)
	
	def OnButtonDeleteClicked(self, *args):
		selection = self.GetSelectedPaths()
		if not selection:
			return
		if len(selection)==1:
			rid, rpath, rtm, revent = selection[0]
			if not Ask(self.window, "You are about to delete '%s'\nfrom %s" % (rpath,time.strftime("%X %x", time.localtime(rtm)))):
				return
		else:
			if not Ask(self.window, "You are about to delete %d files" % (len(selection))):
				return
		
		try:
			self.interface.Delete(selection)
			
			del self.cell[self.selectedTmRng]
			self.UpdateBoxes()
			self.FillTree(self.selectedTmRng)
		except:
			Debug(D_NORM, "Could not delete file\n", printTraceback=True)
		

	def OnButtonRevertClicked(self, *args):
		selection = self.GetSelectedPaths()
		if not selection:
			return
		if len(selection)==1:
			rid,rpath,rtm,revent = selection[0]
			if not Ask(self.window, "You are about to restore '%s'\nfrom %s" % (rpath,time.strftime("%X %x", time.localtime(rtm)))):
				return
			title = 'Please choose the location you wish to restore to'
			suggestedPath = rpath+'.'+time.strftime(DATETIME_STAMP, time.localtime(rtm))
			dlg = gtk.FileChooserDialog(title, None, gtk.FILE_CHOOSER_ACTION_SAVE, 
				(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, gtk.STOCK_SAVE, gtk.RESPONSE_OK))
			dlg.set_do_overwrite_confirmation(True)
			dlg.set_current_name(suggestedPath)
			retVal = dlg.run()
			restorePath = dlg.get_filename()
			dlg.destroy()
			if retVal == gtk.RESPONSE_OK:
				snapshotPath = self.cfg.SnapshotPath(rpath, rtm, self.uid)
				if not os.access(snapshotPath, os.R_OK):
					return
				s = os.stat(snapshotPath)
				shutil.copy2(snapshotPath, restorePath)
				os.chown(restorePath, s.st_uid, s.st_gid)
		else:
			if not Ask(self.window, "You are about to restore %d files" % (len(selection))):
				return
			for rid,rpath,rtm,revent in selection:
				restorePath = rpath+'.'+time.strftime(DATETIME_STAMP, time.localtime(rtm))
				snapshotPath = self.cfg.SnapshotPath(rpath, rtm, self.uid)
				if not os.access(snapshotPath, os.R_OK):
					return
				s = os.stat(snapshotPath)
				shutil.copy2(snapshotPath, restorePath)
				os.chown(restorePath, s.st_uid, s.st_gid)
	
	def Show(self, path='/', tm=time.time(), zoom=ZOOM_HOUR):
		if self.commands.has_key('path'):
			path = self.commands['path']
		
		self.UpdateBoxes(path, tm, zoom)
		self.window.show()
		gtk.main()

	def UpdateBoxes(self, path=None, tm=None, zoom=None):
		if tm:
			self.tm = tm
		if zoom:
			if zoom<ZOOM_MIN_SCALE:
				zoom = ZOOM_MIN_SCALE
			if zoom>=len(ZOOM):
				zoom = len(ZOOM)-1
			self.zoom = zoom
		if path:
			Debug(D_NORM, "Changing path filter to '%s'\n" % path)
			self.path = path
			self.widgets.get_widget("entryFilter").set_text(path)
	
		rng,inc = ZOOM[self.zoom]
		self.tm = int((self.tm+inc-1)/inc)*inc
		
		if self.boxhbox:
			self.cellContainer.remove(self.boxhbox)
		
		self.boxWidth = int(480/(rng/inc))
		self.boxhbox = gtk.HBox()
		for i in range(rng/inc):
			self.boxhbox.pack_end(self.GetCell(self.tm - i*inc, inc), True, True, 1)
		self.cellContainer.add(self.boxhbox)
		self.cellContainer.show_all()
		
		self.listStore.clear()
		self.OnTreeviewRowActivated()
	
	def Destroy(self):
		self.window.destroy()
		sys.exit()
		
	def OnWindowDeleteEvent(self, *args):
		self.Destroy()
		return False
	
	def OnButtonOKClicked(self, *args):
		self.Destroy()
	
	def OnButtonUpClicked(self, *args):
		if self.zoom>=len(ZOOM)-1:
			return
		self.zoom += 1
		centerSelectionTM = self.tm + ZOOM[self.zoom][0]/2
		self.UpdateBoxes(tm=centerSelectionTM)
	
	def OnButtonBackClicked(self, *args):
		self.UpdateBoxes(tm=self.tm-ZOOM[self.zoom][1])
	
	def OnButtonForwardClicked(self, *args):
		self.UpdateBoxes(tm=self.tm+ZOOM[self.zoom][1])
	
	def OnIdleAddRow(self):
		if not self.treeCursor:
			return False
		if not len(self.treeCursor):
			return False			# Done
		
		row = self.treeCursor.pop()
		rid,rtm,rpath,revent,rsz = row
		
		#savedpath = self.cfg.FileStamp(rpath, rtm)		# We want to know about the file at the time it was saved
		#if not os.access(savedpath, os.R_OK):				# Do we have access to read it?
			#return True
		
		# ListStore: 				id, tm(unix), size(B), time, path, event, size
		#							0   1         2        3     4     5      6
		self.listStore.append([rid, rtm, rsz, time.strftime("%x %X",time.localtime(rtm)), rpath, EVENTNAME[revent], HumanSize(rsz)])

		return True
	
	def OnButtonPickDirClicked(self, *args):
		path = SelectDirectory()
		if path:
			self.widgets.get_widget("entryFilter").set_text(path)
			self.OnButtonApplyFilterClicked()
	
	def OnButtonApplyFilterClicked(self, *args):
		path = self.widgets.get_widget("entryFilter").get_text()
		if path!=self.path:
			self.cell = {}
			self.UpdateBoxes(path)
	
	def OnEntryFilterActivate(self, *args):
		self.OnButtonApplyFilterClicked()
	
	def OnCheckbuttonMetaChangeToggled(self, *args):
		self.hideReal = self.widgets.get_widget('checkbuttonMetaChange').get_active()
		
		self.cell = {}
		self.UpdateBoxes()
		if self.selectedRidge:
			self.FillTree(self.selectedTmRng)
	
	def OnCheckbuttonRealChangeToggled(self, *args):
		self.hideMeta = self.widgets.get_widget('checkbuttonRealChange').get_active()
		
		self.cell = {}
		self.UpdateBoxes()
		if self.selectedRidge:
			self.FillTree(self.selectedTmRng)
	
	def OnEventBoxClick(self, widget, signalEvent, tmRng, event):
		if signalEvent.type == gtk.gdk._2BUTTON_PRESS and signalEvent.button == 1:		# Double left-click
			if self.zoom<=ZOOM_MIN_SCALE:
				return
			self.zoom -= 1
			centerSelectionTM = self.cell[tmRng]['start'] + ZOOM[self.zoom][0] - ZOOM[self.zoom][1]
			
			self.UpdateBoxes(tm=centerSelectionTM)
		elif signalEvent.button == 3:		# Right-click
			self.selectedTmRng = tmRng
			self.boxmenu.popup(parent_menu_shell=None, parent_menu_item=None, func=None, button=signalEvent.button, activate_time=signalEvent.time, data=None)
		else:
			self.FillTree(tmRng)
	
	def PackBox(self, tmRng, event, totalPixels, totalSize, vbox):
		sz = self.cell[tmRng][event]

		eb = gtk.EventBox()
		if sz>=1024:
			pixels = int(totalPixels*sz/totalSize)
		else:
			pixels = 0
		eb.set_size_request(-1, pixels)
		eb.modify_bg(gtk.STATE_NORMAL, self.color[event])
		eb.add_events(gtk.gdk.BUTTON_PRESS)
		eb.connect('button_press_event', self.OnEventBoxClick, tmRng, event)
		vbox.pack_start(eb, False, False, 0)
		
	def GetCell(self, tm, span, selected=False):
		tm = int(tm)
		tmRng = "%d-%d" % (tm, tm+span)
		
		eventList = self.EventsOfInterest()
		if not self.cell.has_key(tmRng):
			# Get cell entries
			self.cell[tmRng] = {}
			for event in eventList:
				self.cell[tmRng][event] = 0
				self.cell[tmRng]['#'+event] = 0
			
			ref = self.cell[tmRng]
			Debug(D_VERB, "Getting: %s %d:%d\n" % (tmRng, self.hideMeta, self.hideReal))
			
			try:
				stats = self.interface.Stat(self.path, [tm, tm+span])
				for key in stats:
					ref[key]= int(stats[key])
			except:
				pass				# Empty
			
			ref['start'] = tm
			ref['stop'] = tm+span
		else:
			ref = self.cell[tmRng]
			
		ridge = gtk.EventBox()
		ref['ridge'] = ridge
		if selected:
			ridge.modify_bg(gtk.STATE_NORMAL, self.color['unselected'])
		else:
			self.selectedRidge = ridge
			ridge.modify_bg(gtk.STATE_NORMAL, self.color['unselected'])
		
		vbox = gtk.VBox()
		vbox.set_border_width(1)
		vbox.set_size_request(self.boxWidth, self.boxHeight)
		
		totalCount, totalSize, totalPixels = self.GetCellStats(ref)

		L = gtk.Label("%s\n%d files" % (HumanSize(totalSize), totalCount))
		#L = gtk.Label(HumanSize(totalSize))
		L.set_justify(gtk.JUSTIFY_CENTER)
		L.modify_font(self.smallFont)
		eb = gtk.EventBox()
		eb.connect('button_press_event', self.OnEventBoxClick, tmRng, '')
		eb.add(L)
		vbox.pack_start(eb, True, True, 0)
		
		for event in eventList:
			self.PackBox(tmRng, event, totalPixels, totalSize, vbox)
		ridge.add(vbox)
		
		timeStr = time.strftime("%x\n%X", time.localtime(tm))
		L = gtk.Label(timeStr)
		L.set_justify(gtk.JUSTIFY_CENTER)
		L.modify_font(self.smallFont)
		
		vbox = gtk.VBox()
		vbox.set_border_width(1)
		vbox.pack_start(ridge, False, False, 0)
		vbox.pack_start(L, False, False, 0)
		
		return vbox
