# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import sys
import os
import subprocess
import shutil

import pygtk
pygtk.require("2.0")
import gtk
import gtk.glade

# Add the appropriate path extension so we can import modules without changing directories
sys.path.append(os.path.dirname(__file__))
import config
from base import *
import database
import dbusclient

def Message(window, markup):
	dlg = gtk.MessageDialog(parent=window, flags=gtk.DIALOG_MODAL, type=gtk.MESSAGE_WARNING, buttons=gtk.BUTTONS_OK)
	dlg.set_markup(markup)
	dlg.run()
	dlg.destroy()
	del dlg

def Ask(window, markup):
	dlg = gtk.MessageDialog(parent=window, flags= gtk.DIALOG_MODAL, type=gtk.MESSAGE_WARNING, buttons=gtk.BUTTONS_OK_CANCEL)
	dlg.set_markup(markup)
	ret = dlg.run()
	dlg.destroy()
	del dlg
	if ret==gtk.RESPONSE_OK:
		return True
	return False

def SelectDirectory(title='Please select a directory'):
	dlg = gtk.FileChooserDialog(title, None, gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER, 
		(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, gtk.STOCK_OPEN, gtk.RESPONSE_OK))
	dlg.set_current_folder(os.environ['HOME'])
	
	if dlg.run() == gtk.RESPONSE_OK:
		directory = dlg.get_filename() + '/'	# rsnapshot likes slashes at the end
	else:
		directory = None

	dlg.destroy()
	return directory

def ChooseFile(title='Please select a file', chooseDirectory=False):
	if chooseDirectory:
		action = gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER
	else:
		action = gtk.FILE_CHOOSER_ACTION_OPEN
	
	dialog = gtk.FileChooserDialog(title, None, action, (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, gtk.STOCK_OPEN, gtk.RESPONSE_OK))
	dialog.set_modal(True)
	dialog.set_default_response(gtk.RESPONSE_OK)
	
	selection = None
	response = dialog.run()
	if response == gtk.RESPONSE_OK:
		selection = dialog.get_filename()
	dialog.destroy()
	
	return selection

def Entry(parent=None, title='', markup=''):
	dialog = gtk.Dialog(title=title, parent=parent, flags=0, 
		buttons=(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, gtk.STOCK_OK, gtk.RESPONSE_OK))
	
	label = gtk.Label()
	label.set_markup(markup)
	label.set_line_wrap(True)
	label.set_alignment(xalign=0, yalign=0.5)
	label.set_padding(15,15)
	dialog.vbox.pack_start(label, False, False, 0)
	
	entry = gtk.Entry()
	dialog.vbox.pack_start(entry)
	
	dialog.show_all()
	response = dialog.run()
	dialog.destroy()
	del dialog
	
	if response==gtk.RESPONSE_OK:		# The user clicked the "OK" button
		return entry.get_text()
	return None

'''Main Configuration Window'''
class MainDlg(dbusclient.DBusClient):
	def __init__(self):
		self.commands = ParseCommandLineOptions()
		
		dbusclient.DBusClient.__init__(self)
		self.commands = ParseCommandLineOptions()
		try:
			filename = self.commands['configFile']
		except:
			filename = None
		self.cfg = Configuration(filename)
		
		self.pathsAdded = []
		self.subdirsByPath = {}
	
		# Load the glade file
		gladeFile = config.pkgdatadir+"/config.glade"
		
		self.widgets = gtk.glade.XML(gladeFile)
		self.widgets.signal_autoconnect(self)

		# Connect to main window
		self.window = self.widgets.get_widget("dialogMain")
		self.window.set_size_request(480, -1)
		
		# Prepare the renderers
		textrenderer = gtk.CellRendererText()
  
		# Include Model
		LS = gtk.ListStore(str, str)
		TV = self.widgets.get_widget('treeviewInclude')
		TV.set_model(LS)
		TV.set_headers_visible(True)
		
		# Add the columns to the treeview
		header, col, renderer = 'Path', 0, textrenderer
		column = gtk.TreeViewColumn(header, renderer, text=col)
		column.set_sort_column_id(col)
		column.set_resizable(True)
		TV.append_column(column)
		
		editrenderer = gtk.CellRendererText()
		editrenderer.set_property('editable', True)
		editrenderer.connect("edited", self.CellEdited, LS);
		
		header, col, renderer = 'Delay', 1, editrenderer
		column = gtk.TreeViewColumn(header, renderer, text=col)
		column.set_sort_column_id(col)
		column.set_resizable(True)
		TV.append_column(column)

		self.liststoreInclude = LS
		self.treeviewInclude = TV
		
		# Exclude Model
		LS = gtk.ListStore(str)
		TV = self.widgets.get_widget('treeviewExclude')
		TV.set_model(LS)
		TV.set_headers_visible(True)
		
		# Add the columns to the treeview
		header, col, renderer = 'Pattern', 0, textrenderer
		column = gtk.TreeViewColumn(header, renderer, text=col)
		column.set_sort_column_id(col)
		TV.append_column(column)
		
		self.liststoreExclude = LS
		self.treeviewExclude = TV
	
	def ReloadTrees(self):
		# Sort the backup paths
		self.liststoreInclude.clear()
		for path in self.cfg.settings['backup']:
			entry = self.cfg.settings['backup'][path]
			itt = self.liststoreInclude.append([path, HumanTime(entry['delay'])])

		# Sort the exclusions
		self.liststoreExclude.clear()
		for exp in self.cfg.settings['exclude']:
			self.liststoreExclude.append([exp])
		
	def RefreshView(self):
		self.ReloadTrees()

		root = self.cfg.settings['root']
		if not os.access(root, os.R_OK):
			root = '/'
		
		self.widgets.get_widget("filechooserBackupRoot").set_current_folder(root)
		if self.cfg.settings['enabled']:
			self.widgets.get_widget('checkEnabled').set_active(True)
		if self.cfg.settings['showNotifications']:
			self.widgets.get_widget('checkShowNotifications').set_active(True)
		if self.cfg.settings['nautilusIntegration']:
			self.widgets.get_widget('checkNautilusIntegration').set_active(True)
		if self.cfg.settings['diableAutoMetadata']:
			self.widgets.get_widget('checkDisableAutoMetadata').set_active(True)
		
		hsz = HumanSize(ParseSize(self.cfg.settings['reserved']))
		self.widgets.get_widget('entryMinSpace').set_text(hsz)
		
		hsz = HumanSize(ParseSize(self.cfg.settings['max-size']))
		self.widgets.get_widget('entryMaxSize').set_text(hsz)

		tm = HumanTime(self.cfg.settings['ticInc'])
		self.widgets.get_widget('entryTicInc').set_text(tm)

		self.widgets.get_widget('entryExpireSnaps').set_text(str(self.cfg.settings['retain']['snaps']))
		
		tm = HumanTime(ParseTime(self.cfg.settings['retain']['time']))
		self.widgets.get_widget('entryExpireTime').set_text(tm)

		hsz = HumanSize(ParseSize(self.cfg.settings['retain']['bytes']))
		self.widgets.get_widget('entryExpireBytes').set_text(hsz)
		
		if self.cfg.settings['expire']['meta']:
			self.widgets.get_widget('checkbuttonExpireMeta').set_active(True)

		tm = HumanTime(ParseTime(self.cfg.settings['expire']['merge']))
		self.widgets.get_widget('entryExpireMerge').set_text(tm)
		
		tm = HumanTime(ParseTime(self.cfg.settings['expire']['thin']))
		self.widgets.get_widget('entryExpireThin').set_text(tm)

	def CheckDBVersion(self):
		if self.cfg.settings['configValid']:
			db = database.SqliteDB(self.cfg)
			if not db.Open():
				if Ask(self.window, "The database needs to be upgraded to v.%d (it is currently v.%d). Upgrade?" % (db.CURRENT_DB_VERSION, db.version)):
					db.Upgrade()
	
	def Show(self):
		self.RefreshView()
		self.CheckDBVersion()
		self.window.show()
		gtk.main()

	def Destroy(self):
		self.window.destroy()
		sys.exit()
	
	def ReadBackups(self, model, treePath, treeIter):
		path = self.liststoreInclude.get_value(treeIter, 0)
		delay = self.liststoreInclude.get_value(treeIter, 1)
		if not self.cfg.settings['backup'].has_key(path):
			self.cfg.settings['backup'][path] = {}
		self.cfg.settings['backup'][path]['delay'] = ParseTime(delay)
		
	def Save(self):
		self.cfg.settings['configValid'] = True
		
		self.cfg.settings['enabled'] = self.widgets.get_widget('checkEnabled').get_active()
		self.cfg.settings['showNotifications'] = self.widgets.get_widget('checkShowNotifications').get_active()
		self.cfg.settings['nautilusIntegration'] = self.widgets.get_widget('checkNautilusIntegration').get_active()
		self.cfg.settings['diableAutoMetadata'] = self.widgets.get_widget('checkDisableAutoMetadata').get_active()
		
		self.cfg.settings['root'] = self.widgets.get_widget('filechooserBackupRoot').get_current_folder()+"/"
		self.cfg.settings['reserved'] = self.widgets.get_widget('entryMinSpace').get_text()
		self.cfg.settings['max-size'] = self.widgets.get_widget('entryMaxSize').get_text()
		self.cfg.settings['ticInc'] = ParseTime(self.widgets.get_widget('entryTicInc').get_text())
		if self.cfg.settings['ticInc']<15:
			self.cfg.settings['ticInc'] = 60
		
		self.cfg.settings['retain']['snaps'] = int(self.widgets.get_widget('entryExpireSnaps').get_text())
		self.cfg.settings['retain']['time'] = self.widgets.get_widget('entryExpireTime').get_text()
		self.cfg.settings['retain']['bytes'] = self.widgets.get_widget('entryExpireBytes').get_text()
		self.cfg.settings['expire']['meta'] = self.widgets.get_widget('checkbuttonExpireMeta').get_active()
		self.cfg.settings['expire']['merge'] = self.widgets.get_widget('entryExpireMerge').get_text()
		self.cfg.settings['expire']['thin'] = self.widgets.get_widget('entryExpireThin').get_text()
		
		self.treeviewInclude.get_model().foreach(self.ReadBackups)
		self.cfg.Write()
		self.cfg.MakePaths()

	def OnDialogMainDeleteEvent(self, *args):
		self.Destroy()
		return False

	def OnButtonIncludeAddClicked(self, *args):
		defaultDelay = 60*5
		
		directory = SelectDirectory('Please select a directory to back up')
		if not directory:
			return
		
		if os.path.abspath(directory)=='/':
			Message(self.window, 'TimeVault is not suitable for backing up the root directory. Please individually add the desired backup directories')
			return
		
		if self.cfg.settings['backup'].has_key(directory):
			Message(self.window, "'"+directory+"' " + 'is already in the backup directory list')
		else:
			self.pathsAdded.append(directory)
			itt = self.liststoreInclude.append([directory, HumanTime(defaultDelay)])
				
			self.cfg.settings['backup'][directory] = {'delay': defaultDelay}
	
	def OnButtonIncludeRemoveClicked(self, *args):
		try:
			model,itt = self.treeviewInclude.get_selection().get_selected()		# Returns a (TreeModel, TreeIter) tuple
			selection = self.liststoreInclude.get_value(itt, 0)
			del self.cfg.settings['backup'][selection]
			self.ReloadTrees()
		except:
			pass
	
	def OnButtonBaselineClicked(self, *args):
		try:
			model,itt = self.treeviewInclude.get_selection().get_selected()		# Returns a (TreeModel, TreeIter) tuple
			selection = self.liststoreInclude.get_value(itt, 0)
			path = selection
			if Ask(self.window, "Are you sure you want to take a baseline snapshot of '%s'?" % path):
				db = database.SqliteDB(self.cfg)
				db.Open()
				db.Exec("DELETE FROM sig WHERE path LIKE ?", (path+'%',))
				db.Commit()
				
				self.interface.TakeBaseline(path)
		except:
			Debug(D_NORM, "Error\n", printTraceback=True)
	
	def OnButtonExcludeAddPathClicked(self, *args):
		directory = SelectDirectory('Please select a directory to exclude')
		if not directory:
			return
		
		if directory in self.cfg.settings['exclude']:
			Message(self.window, "'"+directory+"' "+'is already in the exclusion list')
		else:
			self.liststoreExclude.append([directory])
			self.cfg.settings['exclude'].append("%s**" % directory)
	
	def OnButtonExcludeAddRegexClicked(self, *args):
		regex = Entry(self.window, 'RegExp', 'Regular expression to exclude:')
		if not regex:
			return
		
		if regex in self.cfg.settings['exclude']:
			Message(self.window, "'"+regex+"' "+'is already in the exclusion list')
		else:
			self.liststoreExclude.append([regex])
			self.cfg.settings['exclude'].append(regex)
	
	def OnButtonExcludeRemoveClicked(self, *args):
		try:
			model,itt = self.treeviewExclude.get_selection().get_selected()		# Returns a (TreeModel, TreeIter) tuple
			selection = self.liststoreExclude.get_value(itt, 0)
			self.cfg.settings['exclude'].remove(selection)
			self.ReloadTrees()
		except:
			pass

	def OnButtonHelpClicked(self, *args):
		sub = subprocess.Popen(['/usr/bin/gnome-open', 'https://wiki.ubuntu.com/TimeVault'])
		sub.wait()
	
	def OnButtonSaveClicked(self, *args):
		dni = self.widgets.get_widget('checkNautilusIntegration').get_active()
		if dni != self.cfg.settings['nautilusIntegration']:
			#fname = 'timevault-nautilus.py'
			#if dni:
				#shutil.copy2(config.bindir+'/'+fname, NAUTILUS_PLUGIN_PATH+fname)
			#else:
				#try:
					#os.unlink(NAUTILUS_PLUGIN_PATH+fname)
					#os.unlink(NAUTILUS_PLUGIN_PATH+fname+'c')
				#except:
					#pass					# Not there to start with
			#Message(self.window, "You have changed the Nautilus integration settings. These changes will not take effect until you log out.")
			# ToDo: Fix paths so this works again
			Message(self.window, "This functionality is disabled in this version of TimeVault")
		
		enabled = self.widgets.get_widget('checkEnabled').get_active()
		if enabled != self.cfg.settings['enabled']:
			if enabled:
				sub = subprocess.Popen([config.confdir + '/init.d/timevault', 'start'])
				sub.wait()
			else:
				sub = subprocess.Popen([config.confdir + '/init.d/timevault', 'stop'])
				sub.wait()
		
		self.Save()
		
		if len(self.pathsAdded):
			Message(self.window, "You have included new directories in the TimeVault. File signatures will now be computed.")
			for directory in self.pathsAdded:
				self.interface.TakeMeta(directory)
		
		self.Destroy()

	def OnButtonCancelClicked(self, *args):
		self.Destroy()
	
	def OnEntryMaxSizeEditingDone(self, *args):
		want = ParseSize(self.widgets.get_widget('entryMaxSize').get_text())
		self.widgets.get_widget('entryMaxSize').set_text(HumanSize(want))

	def OnEntryMinSpaceEditingDone(self, *args):
		want = ParseSize(self.widgets.get_widget('entryMinSpace').get_text())
		self.widgets.get_widget('entryMinSpace').set_text(HumanSize(want))

	def CellEdited(self, cell, treePath, new_text, listStore):
		self.liststoreInclude[treePath][1] = new_text
		
	def ConnectTreeView(self, name, listStore=None, headers=[""], headersVisible=False):
		if not listStore:
			listStore = gtk.ListStore(str)
		treeView = self.widgets.get_widget(name)
		treeView.set_model(listStore)
		treeView.set_headers_visible(headersVisible)
		
		# Just text rendering for the columns
		textrenderer = gtk.CellRendererText()
		editrenderer = gtk.CellRendererText()
		editrenderer.set_property('editable', True)
		editrenderer.connect("edited", self.CellEdited, listStore);
  
		# Add the columns to the treeview
		i = 0
		for header in headers:
			if i==0:
				column = gtk.TreeViewColumn(header, textrenderer, text=i)
			else:
				column = gtk.TreeViewColumn(header, editrenderer, text=i)
			
			column.set_sort_column_id(i)

			treeView.append_column(column)
			i += 1

		return (listStore, treeView)
