# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

from __future__ import generators
import os.path

from pysqlite2 import dbapi2 as sqlite
import config
from base import *

class Cursor:
	def __init__(self, db):
		self.db = db						# As long as we hang on to this db reference, 
											# it should GC until we have been destroyed
											# hence guaranteeing the ability to commit in 
											# the destructor
		self.cursor = self.db.cursor()
	
	def __del__(self):
		self.Commit()
	
	def Commit(self):
		if self.db:
			self.db.commit()
		
	def Exec(self, *args):
		self.cursor.execute(*args)
		
	def Row(self):
		return self.cursor.fetchone()
	
	def Rows(self, arraysize = 100):
		'''An iterator that uses fetchmany to keep memory usage down'''
		while True:
			results = self.cursor.fetchmany(arraysize)
			if not results:
				break
			for result in results:
				yield result

class SqliteDB:
	CURRENT_DB_VERSION = 5
	
	def __init__(self, cfg):
		self.cfg = cfg
		self.filename = self.cfg.GetDirCatalog() + DB_FILENAME
		self.db = None
		self.version = 0
		
	def __del__(self):
		self.Close()
	
	def Create(self, fromVer=0):
		# We expect to find a db-schema.sql in our directory
		# We're not going to try/except here because if we need to create a 
		# database and the schema doesn't exist, it's over
		sqlDict = open(config.pkgdatadir + "/db-schema.sql", 'r').read()
		sql = ParseRawDataStructsFile(sqlDict)
	
		result = Cursor(self.db)
		for sql in sql[fromVer]:
			Debug(D_NORM, "SQL: %s\n" % sql)
			result.Exec(sql)
		
		result.Commit()
		try:
			self.version = int(self.Exec("SELECT val FROM 'settings' WHERE name='version'").Row()[0])
		except:
			self.version = 0
	
	def Open(self):
		'''Open a connection to a SQLite database and return a cursor'''
		if os.path.exists(self.filename):
			created = False
		else:
			created = True
	
		try:
			Debug(D_VERB, "Opening DB: '%s'\n" % self.filename)
			self.db = sqlite.connect(self.filename)
		except:
			return False		# Directory structure not yet created
		
		# Autocreate only if there is no database present
		if created:
			self.Create(0)
		
		# Check version
		try:
			self.version = int(self.Exec("SELECT val FROM 'settings' WHERE name='version'").Row()[0])
		except:
			self.Create(0)
			
		if self.version!=self.CURRENT_DB_VERSION:
			Debug(D_NORM, "DB: Need v%d, found v%d\n" % (self.CURRENT_DB_VERSION, self.version))
			return False
		
		return True
	
	def Upgrade(self):
		if self.version<self.CURRENT_DB_VERSION:
			for v in range(self.version, self.CURRENT_DB_VERSION):
				Debug(D_NORM, "DB: Upgrading from v%d => v%d\n" % (v, v+1))
				self.Create(v)
	
	def Exec(self, *args):
		c = Cursor(self.db)
		c.Exec(*args)
		return c
		
	def Cursor(self):
		return Cursor(self.db)
	
	def Close(self):
		if self.db:
			self.Commit()
			self.db.close()
		self.db = None

	def Commit(self):
		self.db.commit()
		


#############################################################################################
if __name__ == "__main__":
	cfg = Configuration()
	db = SqliteDB(cfg)
	print("DB at %s, version %d" % (db.filename, db.version))
	results = db.Exec("SELECT name,val FROM settings")
	for row in results.Rows():
		print("%s:\t%s" % (row[0], row[1]))
