# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import gtk

import dbus
import dbus.service
import dbus.glib

'''
Methods:
	GetState()						Returns the current state stack
	GetScheduleTimes()				Returns [(scheduleTime, scheduleNumber)]
	GetScheduleFiles(tm)			Return [(ssID, path, event)]
	AbortSnapshot(path, tm)
	Stat(path, timerange)
	Find(path, timerange)
	Delete(id)						Takes [(id, path, tm, event)]
	TakeBaseline()
	TakeMeta()
	MergeVault(path)
	
Signals:
	OnServerShutdown()				To indicate shutdown
	OnNotifyConfigurationChange()	Indicates Configuration() reload required
	OnState()						State stack has changed
	OnPopup()						Server requests that user be notified
	OnSnap()						Indicates snapshot occurred

StateStack: [[msg, icon]]
TimeRanges: (start, stop) => tm>=start AND tm<stop
'''
class Server(dbus.service.Object):
	def __init__(self):
		self.systemBus = dbus.SystemBus()
		self.busName = dbus.service.BusName("net.launchpad.timevault", bus=self.systemBus)
		dbus.service.Object.__init__(self, self.busName, "/net/launchpad/timevault")

	def __del_(self):
		self.OnServerShutdown()
	
	def GetSender(self, sender):
		if not sender:
			return -1				# Don't return 0, False, or None, so sender cannot possibly compare with root(0)
		return self.systemBus.get_unix_user(sender)
	
	################################################################
	# DBus Methods Follow
	################################################################
	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def Quit(self, sender=None):
		if self.GetSender(sender)==0:
			gtk.main_quit()
			return True
		return False

	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def GetState(self, sender=None):
		# Triggers an OnState
		pass

	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def GetScheduleTimes(self, sender=None):
		# Return [(scheduleTime, scheduleNumber)]
		return []

	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def GetScheduleFiles(self, tm, sender=None):
		# Return [(path, event)]
		return []

	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def AbortSnapshotTm(self, tms, sender=None):
		print(self.GetSender(sender))
	
	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def AbortSnapshotPath(self, paths, sender=None):
		print(self.GetSender(sender))
	
	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def TakeBaseline(self, path, sender=None):
		pass

	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def TakeMeta(self, path, sender=None):
		pass

	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def Stat(self, path, timerange, sender=None):
		pass
	
	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def Find(self, path, timerange, events, sender=None):
		pass
	
	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def Delete(self, deletionList, sender=None):
		pass
	
	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def DeleteQuery(self, path, timerange, events, sender=None):
		pass

	@dbus.service.method("net.launchpad.timevault")
	def Expire(self, ratio, path):
		pass

	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def ClearStatCache(self, tm, sender):
		pass

	@dbus.service.method("net.launchpad.timevault", sender_keyword='sender')
	def ForceSnapshot(self, snaplist, sender=None):
		pass
	
	@dbus.service.method("net.launchpad.timevault")
	def Echo(self, msg):
		self.OnEcho(msg)

	################################################################
	# DBus Signals Follow
	################################################################
	@dbus.service.signal("net.launchpad.timevault")
	def OnEcho(self, msg):
		pass

	@dbus.service.signal("net.launchpad.timevault")
	def OnServerShutdown(self):
		pass

	@dbus.service.signal("net.launchpad.timevault")
	def OnNotifyConfigurationChange(self):
		print("OnNotifyConfigurationChange")
	
	@dbus.service.signal("net.launchpad.timevault")
	def OnPopup(self, summary, markup, duration):
		pass
	
	@dbus.service.signal("net.launchpad.timevault")
	def OnState(self, state):
		# state = [level: (msg, icon)]
		pass
	
	@dbus.service.signal("net.launchpad.timevault")
	def OnScheduleSnap(self, thisTime, thisNum):
		pass
	
	@dbus.service.signal("net.launchpad.timevault")
	def OnSnap(self, tm, num, firstN):
		pass


if __name__ == "__main__":
	dbus_service = Server()
	gtk.main()
