# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import os
import sys
import time

import gobject
import gtk
import gtk.glade
import pygtk
pygtk.require("2.0")

import dbus
import dbus.service
import dbus.glib

import subprocess
import pango

sys.path.append(os.path.dirname(__file__))
import config
from base import *
import dbusclient

DEFAULT_TIMEOUT = 5000
SERVER_KEEPALIVE = 5000

class Pending(dbusclient.DBusClient):
	def __init__(self):
		dbusclient.DBusClient.__init__(self)
		
		self.schedule = []
		self.color = {'B':gtk.gdk.color_parse("blue"), 'N':gtk.gdk.color_parse("darkblue"), 'D':gtk.gdk.color_parse("black"), 
			'C':gtk.gdk.color_parse("green"), 'M':gtk.gdk.color_parse("white"), '#':gtk.gdk.color_parse("red"), 
			'>':gtk.gdk.color_parse("red"), 'selected':gtk.gdk.color_parse("darkgray"), 'unselected':gtk.gdk.color_parse("navajowhite")}
		
		self.smallFont = pango.FontDescription('Sans condensed 6')
		self.mediumFont = pango.FontDescription('Sans 8')
		
		# Load the glade file
		gladeFile = config.pkgdatadir+"/pending.glade"
		
		self.widgets = gtk.glade.XML(gladeFile)
		self.widgets.signal_autoconnect(self)
		
		# Connect to main window
		self.window = self.widgets.get_widget("windowPicker")
		self.window.set_size_request(480,320)
		# Treeview
		headers = ['Time', 'Path', 'Event']
		
		# ListStore:	 tm(unix), size, tm(pretty), event, path, size (pretty)
		#					0   	1		2			3	  4		5
		
		self.treeStore = gtk.TreeStore(int, int, str, str, str, str)
		self.treeView = self.widgets.get_widget('treeviewFiles')
		self.treeView.modify_font(self.mediumFont)
		self.treeView.set_model(self.treeStore)
		self.treeView.set_reorderable(True)
		self.treeView.connect("button-press-event", self.TreeviewClicked, None)
		self.treeView.connect("key-release-event", self.TreeviewKeyPressed, None)
		treeSelection = self.treeView.get_selection()
		treeSelection.set_mode(gtk.SELECTION_MULTIPLE)
		
		# Just text rendering for the columns
		textrenderer = gtk.CellRendererText()
		
		# Add the columns to the treeview
		column = gtk.TreeViewColumn('Time', textrenderer, text=2)
		column.set_sort_column_id(0)
		column.set_resizable(True)
		self.treeView.append_column(column)

		column = gtk.TreeViewColumn('Path', textrenderer, text=3)
		column.set_resizable(True)
		self.treeView.append_column(column)

		column = gtk.TreeViewColumn('Event', textrenderer, text=4)
		column.set_resizable(True)
		self.treeView.append_column(column)
		
		column = gtk.TreeViewColumn('Size', textrenderer, text=5)
		column.set_sort_column_id(1)
		column.set_resizable(True)
		self.treeView.append_column(column)
		
		self.times = {}			# tm: sz, num, parentIter
		self.paths = {}			# path: (tm, event, sz, rowIter)
		self.timesUsed = {}
		self.pathsUsed = {}
		
		self.stale = True
		self.visible = False
		self.destroyed = False
		self.expansion = True
		
		self.MakePopupMenu()
		
	def MakePopupMenu(self):
		self.popupmenu = gtk.Menu()
		
		deli = gtk.ImageMenuItem("Abort")
		deli.connect('activate', self.AbortFile)
		
		self.popupmenu.add(deli)
		self.popupmenu.show_all()
	
	def TreeviewClicked(self, widget, event, data):
		if event.button == 3:		# Right-click
			self.popupmenu.popup(parent_menu_shell=None, parent_menu_item=None, func=None, button=event.button, activate_time=event.time, data=None)

	def TreeviewKeyPressed(self, widget, event, data):
		if event.keyval == gtk.keysyms.Delete:
			pass

	def Show(self):
		self.window.show()
		self.visible = True
		
		gobject.idle_add(self.OnIdleGetSchedule)
		
	def Hide(self):
		self.window.hide()
		self.visible = False
	
	def OnWindowDeleteEvent(self, *args):
		self.Hide()
		self.destroyed = True
		return False
		
	def OnButtonOKClicked(self, *args):
		self.Hide()
	
	def OnButtonRefreshClicked(self, *args):
		gobject.idle_add(self.OnIdleGetSchedule)
	
	def OnNewState(self):
		if self.visible:
			self.OnButtonRefreshClicked()
		else:
			self.stale = True
		
	def OnButtonExpandClicked(self, *args):
		self.treeView.expand_all()

	def OnButtonCollapseClicked(self, *args):
		self.treeView.collapse_all()
	
	def OnButtonDeleteClicked(self, *args):
		self.AbortFile(*args)
		
	def AutoParent(self, tm):
		if self.times.has_key(tm):
			return self.times[tm]
		
		treeIter = self.treeStore.append(None, [tm, 0, time.strftime("%x %X",time.localtime(int(tm))), '', None, ''])
		self.times[tm] = (0, 0, treeIter)
		return self.times[tm]
	
	def UpdateParent(self, tm, sz, num):
		osz, onum, parent = self.AutoParent(tm)
		if sz==osz and num==onum:
			return self.times[tm]
		
		self.treeStore.set_value(parent, 1, sz)
		self.treeStore.set_value(parent, 3, "%d Actions Scheduled" % num)
		self.treeStore.set_value(parent, 5, HumanSize(sz))
		
		return self.times[tm]
	
	def UpdateChild(self, tm, path, event, sz, parent=None):
		if not parent:
			parent = self.AutoParent(tm)
		if self.paths.has_key(path):
			otm, oevent, osz, orowIter = self.paths[path]
			if tm==otm and event==oevent and sz==osz:
				return self.paths[path]
			self.treeStore.remove(orowIter)
		
		rowIter = self.treeStore.append(parent, [tm, sz, None, path, EVENTNAME[event], HumanSize(sz)])
		self.paths[path] = (tm, event, sz, rowIter)
	
	def Purge(self):
		deletable = []
		for path in self.paths:
			if not self.pathsUsed.has_key(path):
				deletable.append(path)
		
		for path in deletable:
			otm, oevent, osz, orowIter = self.paths[path]
			self.treeStore.remove(orowIter)
			del self.paths[path]

		deletable = []
		for tm in self.times:
			if not self.timesUsed.has_key(tm):
				deletable.append(tm)
		
		for tm in deletable:
			osz, onum, parentIter = self.times[tm]
			self.treeStore.remove(parentIter)
			del self.times[tm]
		
	def OnIdleGetSchedule(self):
		self.timesUsed = {}
		self.pathsUsed = {}
		try:
			self.schedule = self.interface.GetScheduleTimes()
			self.schedule.sort()
			self.schedule.reverse()
		except:		# Empty or no server
			self.schedule = []
		
		gobject.idle_add(self.OnIdleGetFiles, self.schedule)
		return False
	
	def OnIdleGetFiles(self, schedule):
		if len(schedule)==0:
			# ToDo: Clean up
			self.Purge()
			return False
		
		now = time.time()
		tm, num = schedule.pop()

		try:
			files = self.interface.GetScheduleFiles(int(tm))
		except:
			return True		# Empty or no server (could have been rescheduled in the last few ms)
		
		totalSize = 0
		fileInfo = []
		self.timesUsed[tm] = True
		for path, event in files:
			try:
				s = os.stat(path)
				sz = s.st_size
				
				totalSize += sz
				fileInfo.append([tm, str(path), str(event), int(sz)])
			except:
				fileInfo.append([tm, str(path), str(event), 0]) 		# File missing (could have been deleted)
			self.pathsUsed[path] = True

		osz, onum, parent = self.UpdateParent(tm, totalSize, num)
		for tm,path,event,sz in fileInfo:
			self.UpdateChild(tm, path, event, sz, parent=parent)

		return True
	
	def GetSelection(self):
		try:
			pathSelection = []
			tmSelection = []
			(model, pathlist) = self.treeView.get_selection().get_selected_rows()
			for treePath in pathlist:
				treeIter = self.treeStore.get_iter(treePath)
				
				tm = self.treeStore.get_value(treeIter, 0)
				path = self.treeStore.get_value(treeIter, 3)
				event = self.treeStore.get_value(treeIter, 4)
				if not event:
					tmSelection.append(tm)
				else:
					pathSelection.append(path)
				
				Debug(D_VERB, "Selection: '%s' [%d]\n" % (path,tm))
			return pathSelection, tmSelection
		except:
			Debug(D_NORM, "No selection\n", printTraceback=True)
			return None, None
		
	def AbortFile(self, *args):
		pathSelection, tmSelection = self.GetSelection()
		if pathSelection and len(pathSelection):
			self.interface.AbortSnapshotPath(pathSelection)
		if tmSelection and len(tmSelection):
			self.interface.AbortSnapshotTm(tmSelection)

class Notifier(dbusclient.DBusClient):
	def __init__(self):
		dbusclient.DBusClient.__init__(self)
		
		self.appName = "TimeVault"
		self.iconFile = config.pix32x32 + '/timevault.png'
		self.InitializeDBusLibNotify()
		
		self.pendingDlg = None
		self.commands = ParseCommandLineOptions()
		
		# ToDo: Cache these in a dictionary to be looked up when needed
		
		self.statusIcon = gtk.StatusIcon()
		self.statusIcon.set_visible(True)
		
		self.statusIcon.connect("popup-menu", self.OnPopupMenu)
		self.statusIcon.connect("activate", self.OnActivate)
		
		self.state = [('', ''), ('', ''), ('', ''), ('Disconnected', TVI_RECONNECT)]
		if self.interface:
			self.interface.GetState()
		if not self.connected:
			self.TryConnect()
			
		self.MakeMenu()
	
	def MakeMenu(self):
		self.menu = gtk.Menu()
		
		self.defaultIcon14x14 = gtk.gdk.pixbuf_new_from_file(config.pix14x14 + '/timevault.png')
		
		selectiImage = gtk.Image()
		selectiImage.set_from_pixbuf(self.defaultIcon14x14)
		selecti = gtk.ImageMenuItem("Snapshot Browser")
		selecti.set_image(selectiImage)
		selecti.connect('activate', self.Activate)
		
		selectRootiImage = gtk.Image()
		selectRootiImage.set_from_pixbuf(self.defaultIcon14x14)
		selectRooti = gtk.ImageMenuItem("Snapshot Browser as Root")
		selectRooti.set_image(selectRootiImage)
		selectRooti.connect('activate', self.ActivateAsRoot)
		
		prefi = gtk.ImageMenuItem(gtk.STOCK_PREFERENCES)
		prefi.connect('activate', self.Pref)
		
		quiti = gtk.ImageMenuItem(gtk.STOCK_QUIT)
		quiti.connect('activate', self.Quit)
		
		self.menu.add(selecti)
		self.menu.add(selectRooti)
		self.menu.add(prefi)
		self.menu.add(gtk.SeparatorMenuItem())
		self.menu.add(quiti)
		self.menu.show_all()
	
	def Quit(self, menuItem):
		sys.exit(0)
	
	def TestServerConnection(self):
		# ToDo: Clean logic since there's no more need to poll for disconnects
		try:
			if self.connected:
				self.interface.GetState()
				Debug(D_NORM, "TestServerConnection() => Connected\n")
			else:
				Debug(D_NORM, 'Diconnected\n')
				self.TryConnect()
		except:
			Debug(D_NORM, 'Shutdown\n')
			self.OnServerShutdown()
		return False
	
	def Pref(self, menuItem):
		sub = subprocess.Popen('gksudo --preserve-env '+config.bindir+'/timevault-config', shell=True, stdout=subprocess.PIPE)
		
	def Activate(self, menuItem):
		sub = subprocess.Popen(config.bindir+'/timevault-browser', shell=True, stdout=subprocess.PIPE)
	
	def ActivateAsRoot(self, menuItem):
		sub = subprocess.Popen('gksudo --preserve-env '+config.bindir+'/timevault-browser', shell=True, stdout=subprocess.PIPE)
	
	def Show(self, summary, markup, msTimeout=DEFAULT_TIMEOUT):
		#UINT32 Notify(STRING app_name, UINT32 replaces_id, STRING app_icon, STRING summary, STRING body, ARRAY actions, DICT hints, INT32 expire_timeout);
		geometry = self.statusIcon.get_geometry()[1]
		self.loc = {"x": geometry.x+5, "y": geometry.y+20}
		
		self.lastMsgId = self.dbus_notify.Notify(self.appName, self.lastMsgId, self.iconFile, summary, markup, ['OK', 'OK'], self.loc, msTimeout)

	def InitializeDBusLibNotify(self):
		bus = dbus.SessionBus()
		bus.get_object("org.freedesktop.DBus","/org/freedesktop/DBus")
		notify_service = bus.get_object("org.freedesktop.Notifications", "/org/freedesktop/Notifications")
		self.dbus_notify = dbus.Interface(notify_service, "org.freedesktop.Notifications")
		
		self.lastMsgId = 0
	
	def OnPopup(self, summary, markup, duration):
		self.Show(summary, markup, duration)
		
	def OnState(self, state):
		Debug(D_NORM, 'OnState()\n')
		level = 0
		for msg, icon in state:
			self.state[level] = (str(msg), str(icon))
			level += 1
		
		msgs = ''
		iconFile = None
		for msg, icon in self.state:
			if len(msg):
				msgs += msg+'\n'
			if len(icon):
				iconFile = icon
		
		if len(msgs) and msgs[-1]=='\n':
			msgs = msgs[:-1]
			
		pix = gtk.gdk.pixbuf_new_from_file(iconFile)
		self.statusIcon.set_from_pixbuf(pix)
		self.statusIcon.set_tooltip(msgs)
		if self.pendingDlg:
			self.pendingDlg.OnNewState()

	def OnScheduleSnap(self, thisTime, thisNum):
		Debug(D_NORM, 'OnScheduleSnap()\n')
		if self.interface:
			self.interface.GetState()
		
		#nextTimeS = time.strftime("%X", time.localtime(nextTime))
		#thisTimeS = time.strftime("%X", time.localtime(thisTime))
		#if thisNum==1:
			#thisPlural = ''
		#else:
			#thisPlural = 's'
		#if nextNum==1:
			#nextPlural = ''
		#else:
			#nextPlural = 's'

		#hint = 'New action: %d file%s at %s\nNext action: %d file%s at %s' % (thisNum, thisPlural, thisTimeS, nextNum, nextPlural, nextTimeS)
		
		#self.statusIcon.set_tooltip(hint)
		#self.statusIcon.set_from_pixbuf(self.pendingIcon)

	def OnSnap(self, tm, num, firstN):
		Debug(D_NORM, 'OnSnap()\n')
		
		if num>0:
			markup = "<small>%d files updated:\n" % int(num)
			rows = firstN.split("\n")
			for row in rows:
				path, event = row.split("\t")
				markup += "<b>%s:</b>%s\n" % (EVENTNAME[event], str(path))
			if len(rows)<num:
				markup += "\t...\n"						# Truncation occurred
			markup += "</small>"
			self.Show("Snapshot", markup, 3000)

		if self.interface:
			self.interface.GetState()

	def TryConnect(self):
		try:
			self.ConnectToServer()
			if self.connected and self.interface:
				if self.pendingDlg:
					self.pendingDlg.ConnectToServer()
				self.interface.GetState()
				return True
		except:
			Debug(D_NORM, "Failed to connect to server\n")
		self.OnServerShutdown()
		return False
	
	def OnServerShutdown(self):
		Debug(D_NORM, "OnServerShutdown()\n")
		self.connected = False
		self.state[TVS_ERROR] = ('Disconnected', TVI_RECONNECT)
		self.OnState(self.state)
		gobject.timeout_add(SERVER_KEEPALIVE, self.TestServerConnection)

	def OnPopupMenu(self, icon, button, activateTime):
		self.menu.popup(parent_menu_shell=None, parent_menu_item=None, func=None, button=button, activate_time=activateTime, data=None)

	def OnActivate(self, icon):
		if not self.pendingDlg:
			self.pendingDlg = Pending()
			
		if self.pendingDlg.destroyed:
			self.pendingDlg = Pending()

		if self.pendingDlg.visible:
			self.pendingDlg.Hide()
		else:
			self.pendingDlg.Show()
