# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import sys
import os
import time

import gobject

import signal

from heapq import heappush

from base import *
from TimeVault import expire
import dbusserver
import watcher
import snap
import database

class TimeVault(dbusserver.Server):
	def __init__(self, commands={}):
		if os.access(DEFAULT_LOCK_FILE, os.R_OK):
			fdLock = open(DEFAULT_LOCK_FILE, "r")
			pid = fdLock.read()
			for line in Exec('ps -p %d -o pid --no-heading' % int(pid)):
				Debug(0, "Found pid %d" % int(line))
				if line.strip()==pid.strip():
					Debug(0, "Another TimeVault server instance is already running\n")
					sys.exit(-1)
			fdLock.close()
		
		fdLock = open(DEFAULT_LOCK_FILE, "w")
		fdLock.write("%d" % os.getpid())
		fdLock.close()
		
		SetFileLogging(DEFAULT_LOG_PATH + 'timevault.log')
		
		dbusserver.Server.__init__(self)
		self.deferTimer = {}
		
		signal.signal(signal.SIGINT, self.OrderlyShutdown)
		signal.signal(signal.SIGTERM, self.OrderlyShutdown)
		
		Debug(D_NORM, "TimeVault v.%s Started: process id=%s, process group=%s\n" % (config.VERSION, os.getpid(), os.getpgrp()))

		self.commands = ParseCommandLineOptions()
		try:
			filename = self.commands['configFile']
		except:
			filename = None
		
		self.started = int(time.time())
		self.cfg = Configuration(filename)
		
		self.pendingFiles = []
		self.pendingDirs = []
		self.pathDict = {}		# [path]=(absTime)
		self.timeDict = {}		# [absTime] = {path1:absTime, ...]
		self.schedule = []
		
		self.notificationTime = 3000
		self.lastBalloonNotification = 0
		self.lastStateNotification = 0
		self.snapshotter = None
		self.expireTimer = None
		self.db = None
		
		self.watcher = watcher.Watcher(self.INotifyCallback, self.cfg)
		self.watcher.watchesRegisteredCallback = self.ReportWatchedDirs
		
		self.Reload(reloadConfiguration=False)
		self.SetState(TVS_IDLE, 'Initializing', TVI_UNCONFIGURED)
		
		if Verbosity()>1:
			gobject.timeout_add(self.cfg.settings['bipTimer'], self.Bip)
		
		self.LoadPendingFiles()
	
	def __del__(self):
		if os.access(DEFAULT_LOCK_FILE, os.R_OK):
			os.remove(DEFAULT_LOCK_FILE)
	
	def Die(self, sig, frame):
		if sig==signal.SIGINT:
			sigName = "SIGINT"
		elif sig==signal.SIGTERM:
			sigName = "SIGTERM"
		else:
			sigName = str(sig)
			
		Debug(D_NORM, "TimeVault v.%s performing orderly shutdown (%s)\n" % (config.VERSION, sigName))
		sys.exit(0)
	
	def OrderlyShutdown(self, sig, frame):
		self.OnServerShutdown()
		gobject.timeout_add(100, self.Die, sig, frame)
	
	def Reload(self, reloadConfiguration=True):
		if reloadConfiguration:
			self.cfg.Reload()
		self.watcher.Clear()
		self.startedWatching = time.time()

		self.state = [('Server running', TVI_UNCONFIGURED), ('', ''), ('', ''), ('', '')]
		self.snapshotNotificationCount = self.cfg.settings['snapshotNotificationCount']
		self.reloadScheduled = None
		self.change = False
		self.deferredCursor = None
		self.watcher.Add(self.cfg.filename, force=True)
		if self.cfg.settings['configValid']:
			self.cfg.MakePaths()
			if not self.db:
				self.db = database.SqliteDB(self.cfg)
				if not self.db.Open():
					# Either wrong version or could not create it
					self.SetState(TVS_WARN, "Database upgrade needed\n\t- Upgrade available in Preferences dialog", TVI_UNCONFIGURED)
					self.db = None
					return False
			
			if self.expireTimer:
				gobject.source_remove(self.expireTimer)
			if self.cfg.settings['expire']['period']:
				self.expireTimer = gobject.timeout_add(self.cfg.settings['expire']['period']*1000, self.Expire, 
					self.cfg.settings['expire']['%'], None)
			
			self.deferredCursor = self.db.Cursor()
			if not self.snapshotter:
				self.snapshotter = snap.Sync(self.cfg, self.db)
			
			self.statCaching = self.cfg.settings['serverStatCachingEnabled']
			self.ticInc = self.cfg.settings['ticInc']
			
			badDirs = []
			for path in self.cfg.settings['backup']:
				if not self.watcher.Add(path):
					badDirs.append(path)
			
			for path in badDirs:
				Debug(D_NORM, "Removing bad dir: '%s'\n" % path)
				del self.cfg.settings['backup'][path]
			
			self.SetState(TVS_WARN, '', '')
		else:
			self.SetState(TVS_WARN, "Unconfigured", TVI_UNCONFIGURED)
		
		# Let everyone know
		self.OnNotifyConfigurationChange()
		return False
	
	def SetState(self, level, text, icon):
		self.state[level] = (text, icon)
		self.Defer(self.DeferredIssueState)
	
	def ReportWatchedDirs(self, finished=False):
		if finished:
			Bench("FinishedWatchingDirs", self.startedWatching, self.watcher.watchedDirs)
			
		if self.watcher:
			self.SetState(TVS_IDLE, "Watching %d directories" % self.watcher.watchedDirs, TVI_NORM)
			if self.watcher.watchedDirs>self.watcher.maxUserWatches:
				self.SetState(TVS_WARN, "Too many directories (%d>%d):\n- Some changes may be lost" % (self.watcher.watchedDirs, self.watcher.maxUserWatches), TVI_ERROR)
	
	def RelativeT(self, t):
		return int(t) - self.started
	
	def Bip(self):
		Debug(D_NORM, ". %d\n" % self.RelativeT(time.time()))
		if self.change:
			self.DumpDictionaries()
			self.change = False
		
		return True
	
	def LoadPendingFiles(self):
		if not self.db:
			return
		
		now = int(time.time())
		queued = 0
		
		rescheduleAggressiveness = self.cfg.settings['rescheduleAggressiveness']
		cursor = self.db.Exec("SELECT tm,path,event FROM pending")
		for tm,path,event in cursor.Rows():
			try:
				pushout = int(queued/rescheduleAggressiveness)*self.ticInc
				if tm<now:
					tm = now+pushout
				else:
					tm = tm+pushout
				
				if not self.pathDict.has_key(path):
					Debug(D_VERB, "Restoring: '%s' [pushed %d]\n" % (path, pushout))
					self.ScheduleSnap(path, event, tm, saveToPendingQueue=False)
					queued += 1
			except:
				Debug(D_NORM, "Unknown wierdness on (%d,%s,%s)\n" % (tm,path,event), printTraceback=True)
		
		Bench("LoadPendingFiles", now, queued)

	def OnTimedProcessPendingDirQueue(self):
		L = len(self.pendingDirs)
		if L==0:
			self.SetState(TVS_INFO, '', '')
			self.pendingCursor = None
			gobject.idle_add(self.LoadPendingFiles)
			return False
			
		now = int(time.time())
		path, event = self.pendingDirs.pop()
		files = os.listdir(path)
		F = len(files)
		
		Debug(D_VERB, "Queuing %s:'%s' [%d files]\n" % (event, path, F))
		for file in files:
			file = os.path.join(path, file)
			if os.path.isdir(file):
				continue
			try:
				Debug(D_VERB, "Queuing '%s' for crawling\n" % (path))
				if not self.pathDict.has_key(file):
					self.AddPendingFile(file, event, now)
			except:
				Debug("Unknown wierdness on (%d,%s,%s)\n" % (file))
		
		Bench("Queue", now, F)
		return True
	
	def OnTimedProcessForcedSnapshots(self, snaplist):
		if not snaplist:
			return False
		
		path, event, tm = snaplist.pop()
		Debug(D_NORM, "Generating %s event on %s at %s\n" % (EVENTNAME[event], path, time.strftime("%X %x", time.localtime(tm))))
		self.ScheduleSnap(path, event, tm, saveToPendingQueue=False)
		return True
	
	def SpiderDirs(self, path, event):
		for root, dirs, files in os.walk(path):
			if self.cfg.Excluded(root):
				continue
			if os.path.isdir(root):
				self.pendingDirs.append((root,event))
		
	def Popup(self, summary, markup):
		self.OnPopup(summary, markup, self.notificationTime)
	
	def DumpDictionaries(self):
		return
	
		msg = "======= DumpDictionaries at t=%d =======\n" % self.RelativeT(time.time())
		msg += "self.timeDict:\n"
		for t in self.timeDict:
			msg += "\t%3d: %s\n" % (self.RelativeT(t), self.timeDict[t])
		msg += "self.pathDict:\n"
		for t in self.pathDict:
			msg += "\t%3s: %s\n" % (t, self.RelativeT(self.pathDict[t]))
		msg += "self.schedule: "
		for t in self.schedule:
			msg += "%3d " % (self.RelativeT(t))
		msg += "\n=========================================\n"
		Debug(D_ALL, msg)
		
	def Defer(self, func, *args):
		# Combines calls made within settings['deferDelay'] into one call
		if not self.deferTimer.has_key(func):
			# defer func prototype: def func(self, func to unset from self.deferTimer, *args)
			self.deferTimer[func] = gobject.timeout_add(self.cfg.settings['deferDelay'], func, func, *args)
	
	def DeferredCommitDB(self, func, *args):
		if self.db:
			self.deferredCursor = self.db.Cursor()
		else:
			self.deferredCursor = None
		
		if self.deferTimer.has_key(func):
			del self.deferTimer[func]
		
		return False
	
	def DeferredBalloonNotify(self, func, absTime, L):
		if self.cfg.settings['showNotifications']:
			self.OnScheduleSnap(absTime, L)
			
		if self.deferTimer.has_key(func):
			del self.deferTimer[func]
		return False
	
	def DeferredIssueState(self, func, *args):
		self.OnState(self.state)
		if self.deferTimer.has_key(func):
			del self.deferTimer[func]
		
		return False
	
	def SetSnapTime(self, absTime):
		if not self.timeDict.has_key(absTime):
			delay = int(absTime - time.time())
			if delay<=0:
				delay = 1
			
			if absTime not in self.schedule:
				heappush(self.schedule, absTime)				# record it in the schedule
																# and set a timer
				gobject.timeout_add(delay*1000, self.TakeSnap, absTime)
			self.timeDict[absTime] = {}							# Create the timeslot

	def RefreshPendingFile(self, path, event, absTime):
		try:
			self.deferredCursor.Exec("UPDATE pending SET tm=?, event=? WHERE path=?", (absTime, event, path))
		except:
			Debug(D_NORM, "Cannot update (%d,%s,%s)\n" % (absTime, path, event), printTraceback=True)
			
			# ToDo: File not inserted; We could correct the error, but for now we let it stick to catch mistakes
			self.AddPendingFile(path, event, absTime)
		
	def AddPendingFile(self, path, event, absTime):
		try:
			self.deferredCursor.Exec("INSERT INTO pending (tm,path,event) VALUES (?,?,?)", (absTime, path, event))
		except:
			Debug(D_NORM, "Cannot insert (%d,%s,%s)\n" % (absTime, path, event), printTraceback=True)
	
	def DelPendingFile(self, path):
		try:
			self.deferredCursor.Exec("DELETE FROM pending WHERE path=?", (path,))
		except:
			Debug(D_NORM, "Cannot delete (%s)\n" % (path), printTraceback=True)
	
	def RoundUp(self, tm):
		return int((tm+self.ticInc-1)/self.ticInc) * self.ticInc
		
	def ScheduleSnap(self, path, event, absTime, saveToPendingQueue=True):
		if self.cfg.Excluded(path):
			self.DelPendingFile(path)
			return
		
		# Move up to the next tic
		Debug(D_VERB, "ScheduleSnap: %s: %d => " % (path, self.RelativeT(absTime)))
		absTime = self.RoundUp(absTime)
		Debug(D_VERB, "%d\n" % self.RelativeT(absTime), timeStamp=False)
		
		self.Defer(self.DeferredCommitDB)
		if self.pathDict.has_key(path):							# Are we rescheduling?
			if saveToPendingQueue:
				self.RefreshPendingFile(path,event,absTime)		# Record the file change so that we can resume if interrupted
			
			schedTime = self.pathDict[path]
			if absTime == schedTime:							# To the same timeslot?
				Debug(D_VERB, "Refreshing %s\n" % path)
				self.timeDict[absTime][path] = event			# If so, update the event and
				return											# we're done
			
			Debug(D_VERB, "Rescheduling %s\n" % path)
			if self.timeDict.has_key(schedTime) and self.timeDict[schedTime].has_key(path):
				del self.timeDict[schedTime][path]				# else, delete it from the last slot
		else:
			if saveToPendingQueue:
				self.AddPendingFile(path,event,absTime)			# Record the file change so that we can resume if interrupted
			
		if not self.timeDict.has_key(absTime):
			self.SetSnapTime(absTime)							# Schedule it
		
		self.pathDict[path] = absTime							# Update/set the path's snap time
		self.timeDict[absTime][path] = event					# and insert the path in the correct timeslot
		self.change = True
		
		self.Defer(self.DeferredBalloonNotify, absTime, len(self.timeDict[absTime]))
	
	def TakeSnap(self, absTime):
		if not self.timeDict.has_key(absTime):
			return False										# Nothing to do
		if absTime not in self.schedule:
			Debug(D_NORM, "Error: TakeSnap scheduled for %d, not in heap\n" % (absTime))
			return
		
		Debug(D_NORM, "Snap@%d: " % (int(time.time())-self.started))
		if not self.timeDict.has_key(absTime):
			Debug(D_NORM, "Nothing to do...\n" % (absTime))
			return False
		
		now = time.time()
		paths = self.timeDict[absTime]
		snapshotsProcessed, snapshotsTaken, snapshotsFailed = self.snapshotter.TakeSnap(paths, absTime)
		for path,event in snapshotsProcessed:
			self.DelPendingFile(path)
			del self.pathDict[path]
		self.db.Commit()
		
		del self.timeDict[absTime]
		self.schedule.remove(absTime)
		self.InvalidateCache('/', absTime)
		
		# DBus Notification
		T = len(snapshotsProcessed)
		L = len(snapshotsTaken)
		F = len(snapshotsFailed)
		Debug(D_NORM, "\t%d processed, %d taken (%d failed)\n" % (T,L,F), timeStamp=False)
		Bench("TakeSnap", now, T)
		if T and self.cfg.settings['showNotifications']:
			firstN = []
			
			# ToDo: Don't show filenames in signal - rather just inform of change and allow retrieval
			# so that access rights can be enforced
			for path,event in snapshotsTaken[:self.snapshotNotificationCount]:
				firstN.append("%s\t%s" % (path, event))
			self.OnSnap(absTime, L, "\n".join(firstN))
		else:
			self.GetState()
		
		self.DumpDictionaries()
		return False
	
	def INotifyCallback(self, path, event):
		if path==self.cfg.filename:
			Debug(D_NORM, "Config File Changed [%s:%s]: Reload Forced\n" % (EVENTNAME[event], path))
			if self.reloadScheduled:
				gobject.source_remove(self.reloadScheduled)
			self.reloadScheduled = gobject.timeout_add(1000, self.Reload)	# we delay a little in case the file was 
																			# subscribed to from diff. locations
		if not self.cfg.settings['configValid']:
			return
		if not self.snapshotter:
			return
		if not self.db:
			return
		if self.cfg.Excluded(path):
			return
		
		longestMatch = self.cfg.LongestPathMatch(path)
		if longestMatch:
			delay = self.cfg.settings['backup'][longestMatch]['delay']
			absTime = int(time.time()+0.999) + delay
			self.ScheduleSnap(path, event, absTime)
	
	def AccessOK(self, path, uid):
		if uid==0:
			return True
		
		try:
			s = os.stat(path)
			if s.st_uid==uid:
				return True
		except:
			# ToDo: Explore failure modes here
			# Remember: What about deleted files
			return True

		return False

	def InvalidateCache(self, path, tm):
		if self.statCaching:
			try:
				self.db.Exec("DELETE FROM statcache WHERE start<=? AND stop>?", (tm, tm))
				Debug(D_VERB, "Invalidated %s from cache: DELETE FROM statcache WHERE start<=%d AND stop>%d\n" % (path, tm, tm))
			except:
				Debug(D_VERB, "Could not delete %s from cache\n" % path, printTraceback=True)
		
	def GetFromStatCache(self, path, start, stop, uid):
		cursor = self.db.Exec("SELECT event, totalsz, totalcount FROM statcache WHERE start=? AND stop=? AND path=?",
				(start, stop, path))
		
		events = {}
		for event, totalsz, totalcount in cursor.Rows():
			events[event] = totalsz
			events['#'+event] = totalcount
		
		if len(events)==0:
			return self.GenerateStats(path, start, stop, uid)
		else:
			Debug(D_VERB, "Cache hit on stats for uid-%d: %s (%d-%d)\n" % (uid, path, start, stop))
		
		return events
			
	def GenerateStats(self, path, start, stop, uid):
		Debug(D_VERB, "Generating Stats for uid-%d: %s (%d-%d)\n" % (uid, path, start, stop))
		
		if uid==0:
			if not path or path=='/':
				rcursor = self.db.Exec("SELECT event, SUM(sz), count(sz) FROM log WHERE tm>=? AND tm<? GROUP BY event",
					(start, stop))
			else:
				rcursor = self.db.Exec("SELECT event, SUM(sz), count(sz) FROM log WHERE tm>=? AND tm<? AND path=? GROUP BY event",
					(start, stop, path))
		else:
			if not path or path=='/':
				rcursor = self.db.Exec("SELECT event, SUM(sz), count(sz) FROM log WHERE tm>=? AND tm<? AND uid=? GROUP BY event",
					(start, stop, uid))
			else:
				rcursor = self.db.Exec("SELECT event, SUM(sz), count(sz) FROM log WHERE tm>=? AND tm<? AND uid=? AND path=? GROUP BY event",
					(start, stop, uid, path))
		
		if self.statCaching:
			wcursor = self.db.Cursor()

		events = {}
		for event, totalsz, totalcount in rcursor.Rows():
			events[event] = totalsz
			events['#'+event] = totalcount
			
			if self.statCaching:
				try:
					wcursor.Exec("INSERT INTO statcache (start, stop, path, uid, event, totalsz, totalcount) VALUES (?,?,?,?,?,?,?)", 
						(start, stop, path, uid, event, totalsz, totalcount))
				except:
					Debug(D_VERB, "Could not insert %s into cache\n" % path, printTraceback=True)
		
		if len(events)==0 and self.statCaching:
			try:
				# We should insert at least one, so we can record the fact that, we have already checked, but there's nothing there
				# Use the special placemarker, '-'
				wcursor.Exec("INSERT INTO statcache (start, stop, path, uid, event, totalsz, totalcount) VALUES (?,?,?,?,'-',0,0)", 
					(start, stop, path, uid))
			except:
				Debug(D_VERB, "Could not insert placemarker %s into cache\n" % path, printTraceback=True)
		
		return events
	
	def ReapChild(self, pid):
		Debug(D_NORM, 'Checking to see if expire process (%d) finished: ' % pid)
		killedpid, stat = os.waitpid(pid, os.WNOHANG)
		if killedpid:
			Debug(D_NORM, 'done - reaped %d\n' % killedpid)
			return False
		
		Debug(D_NORM, 'not finished\n')
		return True
	
	def Run(self):
		gobject.MainLoop().run()

	#########################################################################################
	# DBus Functions																		#
	#########################################################################################
	
	#########################################################################################
	# ToDo: Make very very sure that the sqlite sanitation code has no know vulnerabilities	#
	#########################################################################################
	
	def Find(self, path, timerange, events, sender=None):
		# ToDo: Break up into smaller messages using LIMIT,OFFSET, return cursor
		# Currently, not a very serious issue since it seems that max msg size is
		# DBUS_MAXIMUM_MESSAGE_LENGTH = 128MB
		uid = self.GetSender(sender)
		
		start, stop = timerange
		path = str(path)
		start = int(start)
		stop = int(stop)
		events = str(events)
		
		if events:
			extra = "AND event IN ("
			for e in events:
				extra += "'%s'," % e
			extra = extra[:-1] + ")"
		else:
			extra = ''
		
		start, stop = timerange
		if uid==0:
			cursor = self.db.Exec("SELECT id, tm, path, event, sz FROM log WHERE tm>=? AND tm<? AND path LIKE ? %s ORDER BY tm" % extra,
				(start, stop, '%'+path+'%'))
		else:
			cursor = self.db.Exec("SELECT id, tm, path, event, sz FROM log WHERE tm>=? AND tm<? AND uid=? AND path LIKE ? %s ORDER BY tm" % extra,
				(start, stop, uid, '%'+path+'%'))
		
		rows = []
		for row in cursor.Rows():
			rows.append(row)
		
		return rows
	
	def Stat(self, path, timerange, sender=None):
		uid = self.GetSender(sender)
		
		start, stop = timerange
		path = str(path)
		start = int(start)
		stop = int(stop)
		
		if self.statCaching:
			events = self.GetFromStatCache(path, start, stop, uid)
		else:
			events = self.GenerateStats(path, start, stop, uid)
		
		return events
	
	def Delete(self, deletionList, sender=None):
		uid = self.GetSender(sender)
		cursor = self.db.Cursor()
		for id, origPath, tm, event in deletionList:
			# Note, the event sent may be raw or formatted
			if uid==0:
				cursor.Exec("DELETE FROM log WHERE id=? AND path=? AND tm=?", (id, origPath, tm))
			else:
				cursor.Exec("DELETE FROM log WHERE id=? AND path=? AND tm=? AND uid=?", (id, origPath, tm, uid))
			
			self.InvalidateCache(origPath, tm)
			if event in 'BCN':
				snapshotPath = self.cfg.SnapshotPath(origPath, tm, uid)
				os.unlink(snapshotPath)
		
		del cursor
	
	def DeleteQuery(self, path, timerange, events, sender=None):
		uid = self.GetSender(sender)
		delList = self.Find(path, timerange, events, sender)
		
		cacheInvalidations = {}
		cursor = self.db.Cursor()
		for id, tm, path, event, sz in delList:
			if uid==0:
				cursor.Exec("DELETE FROM log WHERE id=? AND path=? AND tm=?", (id, path, tm))
			else:
				cursor.Exec("DELETE FROM log WHERE id=? AND path=? AND tm=? AND uid=?", (id, path, tm, uid))
			cacheInvalidations[tm] = True
			
			if event in 'BCN':
				snapshotPath = self.cfg.SnapshotPath(path, tm, uid)
				os.unlink(snapshotPath)
		del cursor
		
		for tm in cacheInvalidations:
			self.InvalidateCache('/', tm)
	
	def ClearStatCache(self, tmList, sender):
		uid = self.GetSender(sender)
		# Don't need to check for root uid, since the cache is regenerated as needed
		
		if self.statCaching:
			for tm in tmList:
				self.InvalidateCache('/', tm)
		
	def TakeMeta(self, path, sender=None):
		if self.GetSender(sender)==0:
			self.SpiderDirs(path, 'M')
			self.SetState(TVS_INFO, 'Spidering new directories to retrieve metadata', TVI_PENDING)
			gobject.idle_add(self.OnTimedProcessPendingDirQueue)
		
	def TakeBaseline(self, path, sender=None):
		if self.GetSender(sender)==0:
			self.SpiderDirs(path, 'B')
			self.SetState(TVS_INFO, 'Spidering new directories to create baseline', TVI_PENDING)
			gobject.idle_add(self.OnTimedProcessPendingDirQueue)

	def AbortSnapshotTm(self, tms, sender=None):
		uid = self.GetSender(sender)
		
		for tm in tms:
			Debug(D_VERB, "AbortReq: %s\n" % tm)
			if not self.timeDict.has_key(tm):
				continue
			
			for path in self.timeDict[tm]:
				Debug(D_VERB, "Aborting: %s\n" % path)
				try:
					if not self.AccessOK(path, uid):
						continue
					
					self.DelPendingFile(path)
					if self.pathDict.has_key(path):
						tm = self.pathDict[path]
						del self.pathDict[path]
				except:
					pass				# We tried, maybe it already got snapshotted?
			
			if self.timeDict.has_key(tm):
				del self.timeDict[tm]
				
		self.GetState()
	
	def AbortSnapshotPath(self, paths, sender=None):
		uid = self.GetSender(sender)
		
		for path in paths:
			Debug(D_VERB, "AbortReq: %s\n" % path)
			cursor = self.db.Exec("SELECT tm,path FROM pending WHERE path=?", (str(path),))
			for tm, path in cursor.Rows():
				Debug(D_VERB, "Aborting: %s\n" % path)
				try:
					if not self.AccessOK(path, uid):
						continue
					
					self.DelPendingFile(path)
					if self.pathDict.has_key(path):
						tm = self.pathDict[path]
						del self.pathDict[path]
						if self.timeDict[tm].has_key(path):
							del self.timeDict[tm][path]
				except:
					pass				# We tried, maybe it already got snapshotted?
		self.GetState()
	
	def GetScheduleTimes(self, sender=None):
		uid = self.GetSender(sender)
		
		schedule = []
		for t in self.schedule:
			if self.timeDict.has_key(t):
				schedule.append([t, len(self.timeDict[t])])
		return schedule

	def GetScheduleFiles(self, tm, sender=None):
		uid = self.GetSender(sender)
		
		schedule = []
		if self.timeDict.has_key(tm):
			for path in self.timeDict[tm]:
				if self.AccessOK(path, uid):
					schedule.append([str(path), str(self.timeDict[tm][path])])
		
		return schedule

	def GetState(self, sender=None):
		L = len(self.pathDict)
		if L:
			if L==1:
				plurality = ''
			else:
				plurality = 's'
			self.state[TVS_INFO] = ('%d file%s scheduled' % (L, plurality), TVI_PENDING)
		else:
			self.state[TVS_INFO] = ('', '')
		
		self.OnState(self.state)
		return True
	
	def Expire(self, ratio, path):
		if path:
			extra = ['path="%s"' % path]
		else:
			extra = []
		
		pid = subprocess.Popen([config.bindir+'/timevault-expire', '--verbose', '--ratio=%f' % ratio] + extra).pid
		Debug(D_VERB, "Launched EXPIRE module; pid %d\n" % pid)
		gobject.timeout_add(2500, self.ReapChild, pid)
		
		return True
	
	def ForceSnapshot(self, snaplist, sender=None):
		uid = self.GetSender(sender)
		
		if uid==0:
			gobject.idle_add(self.OnTimedProcessForcedSnapshots, snaplist)
		return True
