# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import sys
import os
import shutil
import time
import random
try:
	import hashlib			# New in Python 2.5
except:
	import sha				# Deprecated method

# Add the appropriate path extension so we can import modules without changing directories
sys.path.append(os.path.dirname(__file__))
from base import *
import database

MAX_HASH_DICT_SIZE = 4*1024

class Sync:
	def __init__(self, cfg, db):
		self.cfg = cfg
		self.cfg.MakePaths()
		self.db = db

		# Add --max-size= limitation
		self.maxsize = ParseSize(self.cfg.settings['max-size'])
		
		random.seed()
		self.fileHash = {}
	
	def ComputeFileHash(self, path, s):
		fd = os.open(path, os.O_RDONLY)
		sz = s.st_size
		
		# ToDo: Is it worth taking a new snapshot if the perms or owner:group change?
		try:
			m = hashlib.sha1()
		except:
			m = sha.new()
			
		while True:
			buf = os.read(fd, BLOCKSIZE)
			if not buf:
				break
			m.update(buf)
		os.close(fd)
		return m.hexdigest()
	
	def TrimHashDict(self, trimTo):
		Debug(D_NORM, "Trimming File-Hash dictionary to free RAM (len=%d) => " % len(self.fileHash))
		toTrim = []
		for path in self.fileHash:
			if random.random()<=trimTo:
				toTrim.append(path)
		for path in toTrim:
			del self.fileHash[path]
		Debug(D_NORM, "(len=%d)\n" % len(self.fileHash), timeStamp=False)
		
	def FileHashChanged(self, path, s, cursor):
		currentFileHash = self.ComputeFileHash(path, s)
		
		if self.fileHash.has_key(path):
			prevFileHash = self.fileHash[path]
		else:
			try:
				prevFileHash = self.db.Exec("SELECT hash FROM sig WHERE path=? LIMIT 1", (path,)).Row()[0]
				self.fileHash[path] = currentFileHash
				if currentFileHash!=prevFileHash:
					cursor.Exec("UPDATE sig SET hash=? WHERE path=? LIMIT 1", (currentFileHash, path))
			except:
				self.fileHash[path] = currentFileHash
				prevFileHash = currentFileHash
				cursor.Exec("INSERT INTO sig (path,hash) VALUES (?,?)", (path, self.fileHash[path]))
				return True					# Obviously changed
		
		if len(self.fileHash)>MAX_HASH_DICT_SIZE:
			self.TrimHashDict(0.25)
		
		# Rather than commit locally, we let the calling function do so (when cursor is destroyed)
		return currentFileHash!=prevFileHash
		
	def TakeSnap(self, paths, unixTime):
		pythonTime = time.localtime(unixTime)
		usableSpace = self.cfg.UsableSpaceOnBackupDrive()
		snapshotsProcessed = []
		snapshotsTaken = []
		snapshotsFailed = []
		
		cursor = self.db.Cursor()
		for path in paths:
			event = paths[path]
			try:
				Debug(D_VERB, "Processing: [%s] %s\n" % (event,path))
				snapshotsProcessed.append([path,event])
				
				s = os.stat(path)
				try:
					if os.path.isdir(path):
						cursor.Exec("INSERT INTO log (tm,path,event,sz,uid) VALUES (?,?,?,?,?)", (unixTime, path, event, 0, s.st_uid))
						snapshotsTaken.append([path,event])
						continue
				except:
					# File/Directory was deleted before we could get to it
					pass
					
				if event=='D':
					cursor.Exec("INSERT INTO log (tm,path,event,sz,uid) VALUES (?,?,?,?,?)", (unixTime, path, event, 0, s.st_uid))
					snapshotsTaken.append([path,event])
					continue
				
				sz = s.st_size
				if sz>usableSpace:
					event = '#'			# Emit error?
					Debug(D_NORM, "Not enough space (%s required, %s avail.): %s\n" % (HumanSize(sz), HumanSize(usableSpace), path))
				if sz>self.maxsize:
					event = '>'			# Emit error?
					Debug(D_NORM, "File too large (%s required, maxsize %s): %s\n" % (HumanSize(sz), HumanSize(self.maxsize), path))
				
				if event not in 'D#>':
					if not self.FileHashChanged(path, s, cursor):
						event = 'M'			# Meta-data change only
				
				sql = "INSERT INTO log (tm,path,event,sz,uid) VALUES (%d,'%s','%s',%d,%d)" % (unixTime, path, event, sz, s.st_uid)
				Debug(D_ALL, "SQL: %s\n" % sql)
				cursor.Exec("INSERT INTO log (tm,path,event,sz,uid) VALUES (?,?,?,?,?)", (unixTime, path, event, sz, s.st_uid))
				if event in 'BCN':
					
					# ToDo: clean up after verifying correctness
					userDir = self.cfg.GetDirUser(s.st_uid)
					self.cfg.CheckAndMake(userDir, 0700, s.st_uid)
					destDir = self.cfg.GetDirSnapDest(s.st_uid, pythonTime)
					self.cfg.CheckAndMake(destDir, 0700, s.st_uid)
					
					dest = destDir + self.cfg.FileStamp(path, unixTime)
					shutil.copy2(path, dest)
					os.chown(dest, s.st_uid, s.st_gid)
					
					usableSpace -= sz
					snapshotsTaken.append([path,event])
			except OSError:
				snapshotsFailed.append([path,event])
		
		return snapshotsProcessed, snapshotsTaken, snapshotsFailed

if __name__ == "__main__":
	cfg = Configuration()
	db = database.SqliteDB(cfg)
	s = Sync(cfg, db)

