# -*- coding: UTF-8 -*-
#   TimeVault - automated file backup and restore
#   Copyright (C) 2007 A. Bashi <sourcecontact@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License
#   as published by the Free Software Foundation; either version 2
#   of the License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import sys

import os
import gobject
import gamin

sys.path.append(os.path.dirname(__file__))
from base import *

MAX_USER_WATCH = '/proc/sys/fs/inotify/max_user_watches'

#F_MOVE = gamin.GAMMoved
F_CHANGE = gamin.GAMChanged
F_NEW = gamin.GAMCreated
F_DELETE = gamin.GAMDeleted
F_EXISTS = gamin.GAMExists

class Watcher:
	events = {F_CHANGE: "C", F_NEW: "N", F_DELETE: "D"}
	# ToDo: Does not correctly handle directory moves - currently wastes space
	
	def __init__(self, callback, cfg):
		self.watched = {}
		self.callback = callback
		self.watchesRegisteredCallback = self.NullCallback
		self.cfg = cfg
		self.sid = None
		self.monitor = None
		self.pendingDirs = []
		
		try:
			fd = open(MAX_USER_WATCH, "r")
			self.maxUserWatches = int(fd.read())
			fd.close()
		except:
			self.maxUserWatches = 0				# inotify not installed?
		
		self.Clear()
	
	def __del__(self):
		self.Clear()
	
	def NullCallback(self):
		pass
	
	def Clear(self):
		if self.sid:
			gobject.source_remove(self.sid)
		if self.monitor:
			del self.monitor
		
		self.watched = {}
		self.watchedFiles = 0
		self.watchedDirs = 0
		self.pendingDirs = []
		
		self.monitor = gamin.WatchMonitor()
		self.monitor.no_exists()
		self.fd = self.monitor.get_fd()
		self.sid = gobject.io_add_watch(self.fd, gobject.IO_IN, self.EventsPending)
		
	def EventsPending(self, source, condition):
		self.monitor.handle_events()
		return True
	
	def HandleEvent(self, name, event, directory):
		if self.events.has_key(event):
			path = os.path.join(directory, name)
			
			if event==F_NEW and os.path.isdir(path):
				# Recursively add all the files and dirs
				for root, dirs, files in os.walk(path):
					for name in dirs:
						file = os.path.join(root, name)
						if os.path.isdir(file):
							self.pendingDirs.append((root, False))
					
					for name in files:
						file = os.path.join(root, name)
						if self.cfg.Excluded(file):
							continue
						if not os.path.isdir(file):
							self.callback(file, self.events[F_NEW])
				if len(self.pendingDirs)>0:
					gobject.idle_add(self.OnIdle)
			
			if event==F_CHANGE and os.path.isdir(path):
				return			# We don't care about changed directories
		
			self.callback(path, self.events[event])
	
	def OnIdle(self):
		UPDATE_FREQ = 256
		
		if len(self.pendingDirs)==0:
			self.watchesRegisteredCallback(finished=True)
			return False
		
		# Have at least one, and the report every UPDATE_FREQ
		if (len(self.pendingDirs)%UPDATE_FREQ)==1:
			self.watchesRegisteredCallback(finished=False)

		path,force = self.pendingDirs.pop()
		self.AddSingleDirectory(path, force)
		return True

	def AddSingleDirectory(self, path, force=False):
		if self.watched.has_key(path):
			return True
		
		if force or not self.cfg.Excluded(path):
			self.watchedDirs += 1
			self.watched[path] = True
			self.monitor.watch_directory(path, self.HandleEvent, path)
			Debug(D_ALL, "Watching: %s\n" % path)
			
			return True
		
		return False
	
	def Add(self, path, force=False):
		if not os.access(path, os.R_OK):
			return False
		
		try:
			if os.path.isdir(path):
				for root, dirs, files in os.walk(path):
					# ToDo: Create a manual walk function so that exclusions can be trimmed early
					# 		In order for that to be possible, exclusion language must be broken down
					#		into file and path exclusions, where path *must be* recursive
					self.pendingDirs.append((root, force))
					
				Debug(D_VERB, "Watching: %s [%d]\n" % (path, len(self.pendingDirs)))
				if len(self.pendingDirs)>0:
					gobject.idle_add(self.OnIdle)
			else:
				if force or not self.cfg.Excluded(path):
					self.watchedFiles += 1
					self.monitor.watch_file(path, self.HandleEvent, os.path.dirname(path))
		except:
			return False
		return True
	
	def Remove(self, path):
		if self.watched.has_key(path):
			del self.watched[path]

		self.monitor.stop_watch(path)

if __name__ == "__main__":
	def callback(path, event):
		print("callback: %s:%s" % (path,event))

	cfg = Configuration()
	w = Watcher(callback, cfg)
	for path in cfg.settings['backup']:
		w.Add(path)
	print(w.watched)
	try:
		gobject.MainLoop().run()
	except KeyboardInterrupt:
		sys.exit(0)
